-------------------------------------------------------------------------------
-- (C) Altran Praxis Limited
-------------------------------------------------------------------------------
--
-- The SPARK toolset is free software; you can redistribute it and/or modify it
-- under terms of the GNU General Public License as published by the Free
-- Software Foundation; either version 3, or (at your option) any later
-- version. The SPARK toolset is distributed in the hope that it will be
-- useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
-- MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General
-- Public License for more details. You should have received a copy of the GNU
-- General Public License distributed with the SPARK toolset; see file
-- COPYING3. If not, go to http://www.gnu.org/licenses for a complete copy of
-- the license.
--
--=============================================================================

package body CStacks is

   --------------------------------------------------------------------------
   -- Reresentation of a Cell Stack
   --
   -- A Stack is a singly linked list of Cells.  In each Cell:
   --
   -- A_Ptr field links the stack together
   -- C_Ptr field points to the data Cell of each stack element
   --------------------------------------------------------------------------

   function IsEmpty (S : Stack) return Boolean is
   begin
      return Cells.Is_Null_Cell (Cells.Cell (S));
   end IsEmpty;

   --------------------------------------------------------------------------

   function Top (Heap : Cells.Heap_Record;
                 S    : Stack) return Cells.Cell is
   begin
      return Cells.Get_C_Ptr (Heap, Cells.Cell (S));
   end Top;

   --------------------------------------------------------------------------

   procedure CreateStack (S : out Stack) is
   begin
      S := Stack (Cells.Null_Cell);
   end CreateStack;

   --------------------------------------------------------------------------

   procedure Pop (Heap : in out Cells.Heap_Record;
                  S    : in out Stack) is
      OldTopCell : Cells.Cell;
   begin
      OldTopCell := Cells.Cell (S);
      S          := Stack (Cells.Get_A_Ptr (Heap, OldTopCell));
      Cells.Dispose_Of_Cell (Heap, OldTopCell);
   end Pop;

   --------------------------------------------------------------------------

   -- New procedure (shorthand for Top followed by Pop)
   procedure PopOff (Heap : in out Cells.Heap_Record;
                     S    : in out Stack;
                     C    :    out Cells.Cell) is
   begin
      C := Top (Heap, S);
      Pop (Heap, S);
   end PopOff;

   --------------------------------------------------------------------------

   procedure Push (Heap     : in out Cells.Heap_Record;
                   CellName : in     Cells.Cell;
                   S        : in out Stack) is
      NewTopCell : Cells.Cell;
   begin
      Cells.Create_Cell (Heap, NewTopCell);
      Cells.Set_A_Ptr (Heap, NewTopCell, Cells.Cell (S));
      Cells.Set_C_Ptr (Heap, NewTopCell, CellName);
      S := Stack (NewTopCell);
   end Push;

   --------------------------------------------------------------------------

   function NonDestructivePop (Heap : Cells.Heap_Record;
                               S    : Stack) return Stack is
   begin
      return Stack (Cells.Get_A_Ptr (Heap, Cells.Cell (S)));
   end NonDestructivePop;

   --------------------------------------------------------------------------

   function FindAggregateCell (Heap : Cells.Heap_Record;
                               S    : Stack) return Cells.Cell is
      Ptr  : Cells.Cell;
      Kind : Cells.Cell_Kind;

   begin
      Ptr := Cells.Cell (S);
      loop
         Kind := Cells.Get_Kind (Heap, Cells.Get_C_Ptr (Heap, Ptr));
         exit when Kind = Cell_Storage.Incomplete_Aggregate;
         Ptr := Cells.Get_A_Ptr (Heap, Ptr);
      end loop;
      Ptr := Cells.Get_C_Ptr (Heap, Ptr);
      return Ptr;
   end FindAggregateCell;

   --------------------------------------------------------------------------

   function FindAggregateCounter (Heap : Cells.Heap_Record;
                                  S    : Stack) return Cells.Cell is
      Ptr : Cells.Cell;
   begin
      Ptr := Cells.Cell (S);
      loop
         exit when Cells.Get_Kind (Heap, Cells.Get_C_Ptr (Heap, Ptr)) = Cell_Storage.Aggregate_Counter;
         Ptr := Cells.Get_A_Ptr (Heap, Ptr);
      end loop;
      Ptr := Cells.Get_C_Ptr (Heap, Ptr);
      return Ptr;
   end FindAggregateCounter;

end CStacks;
