% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/check_predictions.R
\name{check_predictions}
\alias{check_predictions}
\alias{check_predictions.default}
\title{Posterior predictive checks}
\usage{
check_predictions(object, ...)

\method{check_predictions}{default}(
  object,
  iterations = 50,
  check_range = FALSE,
  re_formula = NULL,
  bandwidth = "nrd",
  type = "density",
  verbose = TRUE,
  ...
)
}
\arguments{
\item{object}{A statistical model.}

\item{...}{Passed down to \code{simulate()}.}

\item{iterations}{The number of draws to simulate/bootstrap.}

\item{check_range}{Logical, if \code{TRUE}, includes a plot with the minimum
value of the original response against the minimum values of the replicated
responses, and the same for the maximum value. This plot helps judging whether
the variation in the original data is captured by the model or not
(\emph{Gelman et al. 2020, pp.163}). The minimum and maximum values of \code{y} should
be inside the range of the related minimum and maximum values of \code{yrep}.}

\item{re_formula}{Formula containing group-level effects (random effects) to
be considered in the simulated data. If \code{NULL} (default), condition
on all random effects. If \code{NA} or \code{~0}, condition on no random
effects. See \code{simulate()} in \strong{lme4}.}

\item{bandwidth}{A character string indicating the smoothing bandwidth to
be used. Unlike \code{stats::density()}, which used \code{"nrd0"} as default, the
default used here is \code{"nrd"} (which seems to give more plausible results
for non-Gaussian models). When problems with plotting occur, try to change
to a different value.}

\item{type}{Plot type for the posterior predictive checks plot. Can be \code{"density"},
\code{"discrete_dots"}, \code{"discrete_interval"} or \code{"discrete_both"} (the \verb{discrete_*}
options are appropriate for models with discrete - binary, integer or ordinal
etc. - outcomes).}

\item{verbose}{Toggle warnings.}
}
\value{
A data frame of simulated responses and the original response vector.
}
\description{
Posterior predictive checks mean "simulating replicated data
under the fitted model and then comparing these to the observed data"
(\emph{Gelman and Hill, 2007, p. 158}). Posterior predictive checks
can be used to "look for systematic discrepancies between real and
simulated data" (\emph{Gelman et al. 2014, p. 169}).

\strong{performance} provides posterior predictive check methods for a variety
of frequentist models (e.g., \code{lm}, \code{merMod}, \code{glmmTMB}, ...). For Bayesian
models, the model is passed to \code{\link[bayesplot:pp_check]{bayesplot::pp_check()}}.

If \code{check_predictions()} doesn't work as expected, try setting
\code{verbose = TRUE} to get hints about possible problems.
}
\details{
An example how posterior predictive checks can also be used for model
comparison is Figure 6 from \emph{Gabry et al. 2019, Figure 6}.

\if{html}{\cr \figure{pp_check.png}{options: width="90\%" alt="Posterior Predictive Check"} \cr}
The model shown in the right panel (b) can simulate new data that are more
similar to the observed outcome than the model in the left panel (a). Thus,
model (b) is likely to be preferred over model (a).
}
\note{
Every model object that has a \code{simulate()}-method should work with
\code{check_predictions()}. On R 3.6.0 and higher, if \strong{bayesplot} (or a
package that imports \strong{bayesplot} such as \strong{rstanarm} or \strong{brms})
is loaded, \code{pp_check()} is also available as an alias for \code{check_predictions()}.

If \code{check_predictions()} doesn't work as expected, try setting \code{verbose = TRUE}
to get hints about possible problems.
}
\examples{
\dontshow{if (insight::check_if_installed("see", minimum_version = "0.9.1", quietly = TRUE)) (if (getRversion() >= "3.4") withAutoprint else force)(\{ # examplesIf}
# linear model
model <- lm(mpg ~ disp, data = mtcars)
check_predictions(model)

# discrete/integer outcome
set.seed(99)
d <- iris
d$skewed <- rpois(150, 1)
model <- glm(
  skewed ~ Species + Petal.Length + Petal.Width,
  family = poisson(),
  data = d
)
check_predictions(model, type = "discrete_both")
\dontshow{\}) # examplesIf}
}
\references{
\itemize{
\item Gabry, J., Simpson, D., Vehtari, A., Betancourt, M., and Gelman, A. (2019).
Visualization in Bayesian workflow. Journal of the Royal Statistical Society:
Series A (Statistics in Society), 182(2), 389–402. https://doi.org/10.1111/rssa.12378
\item Gelman, A., and Hill, J. (2007). Data analysis using regression and
multilevel/hierarchical models. Cambridge; New York: Cambridge University Press.
\item Gelman, A., Carlin, J. B., Stern, H. S., Dunson, D. B., Vehtari, A., and
Rubin, D. B. (2014). Bayesian data analysis. (Third edition). CRC Press.
\item Gelman, A., Hill, J., and Vehtari, A. (2020). Regression and Other Stories.
Cambridge University Press.
}
}
\seealso{
\code{\link[=simulate_residuals]{simulate_residuals()}} and \code{\link[=check_residuals]{check_residuals()}}. See also
\code{\link[see:print.see_performance_pp_check]{see::print.see_performance_pp_check()}} for options to customize the plot.

Other functions to check model assumptions and and assess model quality: 
\code{\link{check_autocorrelation}()},
\code{\link{check_collinearity}()},
\code{\link{check_convergence}()},
\code{\link{check_heteroscedasticity}()},
\code{\link{check_homogeneity}()},
\code{\link{check_model}()},
\code{\link{check_outliers}()},
\code{\link{check_overdispersion}()},
\code{\link{check_singularity}()},
\code{\link{check_zeroinflation}()}
}
\concept{functions to check model assumptions and and assess model quality}
