/*
 *  Copyright (c) 2005 Adrian Page <adrian@pagenet.plus.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public License
 * along with this library; see the file COPYING.LIB.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#ifndef GTL_INTEGER_MATHS_H
#define GTL_INTEGER_MATHS_H

#include "StdTypes.h"

namespace GTLCore {

  template<typename _T_, typename _T2_, typename _T3_>
  inline _T_ bound(_T2_ l, _T_ x, _T3_ u)
  {
      if (x < l)
          return _T_(l);
      else if (x > u)
          return _T_(u);
      return x;
  }

  /// take a and scale it up by 256*b/255
  inline unsigned int UINT8_SCALEBY(unsigned int a, unsigned int b)
  {
      unsigned int c = a * b + 0x80u;
      return (c >> 8) + c;
  }

  /// multiplication of two scale values
  /// A scale value is interpreted as 255 equaling 1.0 (such as seen in rgb8 triplets)
  /// thus "255*255=255" because 1.0*1.0=1.0
  inline unsigned int UINT8_MULT(unsigned int a, unsigned int b)
  {
      unsigned int c = a * b + 0x80u;
      return ((c >> 8) + c) >> 8;
  }

  inline unsigned int UINT8_DIVIDE(unsigned int a, unsigned int b)
  {
      unsigned int c = (a * UINT8_MAX + (b / 2u)) / b;
      return c;
  }

  /// Approximation of (a * b * c + 32512) / 65025.0
  inline unsigned int UINT8_MULT3(unsigned int a, unsigned int b, unsigned int c)
  {
    unsigned int t = a * b * c + 0x7F5B;
    return ((t >> 7) + t) >> 16;
  }

  /// Blending of two scale values as described by the alpha scale value
  /// A scale value is interpreted as 255 equaling 1.0 (such as seen in rgb8 triplets)
  /// Basically we do: a*alpha + b*(1-alpha)
  inline unsigned int UINT8_BLEND(unsigned int a, unsigned int b, unsigned int alpha)
  {
      // However the formula is refactored to (a-b)*alpha + b  since that saves a multiplication
      // Signed arithmetic is needed since a-b might be negative
      int c = (int(a) - int(b)) * alpha + 0x80u;
      c = ((c >> 8) + c) >> 8;
      return c + b;
  }

  inline unsigned int UINT16_MULT(unsigned int a, unsigned int b)
  {
      unsigned int c = a * b + 0x8000u;
      return ((c >> 16) + c) >> 16;
  }

  inline int INT16_MULT(int a, int b)
  {
      return (a*b) / INT16_MAX;
  }

  inline unsigned int UINT16_DIVIDE(unsigned int a, unsigned int b)
  {
      unsigned int c = (a * UINT16_MAX + (b / 2u)) / b;
      return c;
  }

  inline unsigned int UINT16_BLEND(unsigned int a, unsigned int b, unsigned int alpha)
  {
      // Basically we do a*alpha + b*(1-alpha)
      // However refactored to (a-b)*alpha + b  since that saves a multiplication
      // Signed arithmetic is needed since a-b might be negative
      int c = ((int(a) - int(b)) * int(alpha)) >> 16;
      return (unsigned int)(c + b);
  }

  inline unsigned int UINT8_TO_UINT16(unsigned int c)
  {
      return c | (c << 8);
  }

  inline unsigned int UINT16_TO_UINT8(unsigned int c)
  {
      //return round(c / 257.0);
      //For all UINT16 this calculation is the same and a lot faster (off by c/65656 which for every c is 0)
      c = c - (c >> 8) + 128;
      return c >> 8;
  }

  inline int INT16_BLEND(int a, int b, unsigned int alpha)
  {
      // Basically we do a*alpha + b*(1-alpha)
      // However refactored to (a-b)*alpha + b  since that saves a multiplication
      int c = ((int(a) - int(b)) * int(alpha)) >> 16;
      return c + b;
  }
}

#endif

