// Licensed to the .NET Foundation under one or more agreements.
// The .NET Foundation licenses this file to you under the MIT license.
// See the LICENSE file in the project root for more information.

using Test.Cryptography;

namespace System.Security.Cryptography.Pkcs.Tests
{
    internal static class Certificates
    {
        public static readonly CertLoader RSAKeyTransfer1 = new CertLoaderFromRawData(RawData.s_RSAKeyTransfer1Cer, RawData.s_RSAKeyTransfer1Pfx, "1111");
        public static readonly CertLoader RSAKeyTransfer2 = new CertLoaderFromRawData(RawData.s_RSAKeyTransfer2Cer, RawData.s_RSAKeyTransfer2Pfx, "1111");
        public static readonly CertLoader RSAKeyTransfer3 = new CertLoaderFromRawData(RawData.s_RSAKeyTransfer3Cer, RawData.s_RSAKeyTransfer3Pfx, "1111");
        public static readonly CertLoader RSAKeyTransfer_ExplicitSki = new CertLoaderFromRawData(RawData.s_RSAKeyTransferCer_ExplicitSki, RawData.s_RSAKeyTransferPfx_ExplicitSki, "1111");
        public static readonly CertLoader RSAKeyTransferCapi1 = new CertLoaderFromRawData(RawData.s_RSAKeyTransferCapi1Cer, RawData.s_RSAKeyTransferCapi1Pfx, "1111");
        public static readonly CertLoader RSASha256KeyTransfer1 = new CertLoaderFromRawData(RawData.s_RSASha256KeyTransfer1Cer, RawData.s_RSASha256KeyTransfer1Pfx, "1111");
        public static readonly CertLoader RSASha384KeyTransfer1 = new CertLoaderFromRawData(RawData.s_RSASha384KeyTransfer1Cer, RawData.s_RSASha384KeyTransfer1Pfx, "1111");
        public static readonly CertLoader RSASha512KeyTransfer1 = new CertLoaderFromRawData(RawData.s_RSASha512KeyTransfer1Cer, RawData.s_RSASha512KeyTransfer1Pfx, "1111");
        public static readonly CertLoader DHKeyAgree1 = new CertLoaderFromRawData(RawData.s_DHKeyAgree1Cer);
        public static readonly CertLoader RSA2048SignatureOnly = new CertLoaderFromRawData(RawData.s_Rsa2048SignatureOnlyCer, RawData.s_Rsa2048SignatureOnlyPfx, "12345");
        public static readonly CertLoader Dsa1024 = new CertLoaderFromRawData(RawData.s_dsa1024Cert, RawData.s_dsa1024Pfx, "1234");
        public static readonly CertLoader ECDsaP256Win = new CertLoaderFromRawData(RawData.ECDsaP256_DigitalSignature_Cert, RawData.ECDsaP256_DigitalSignature_Pfx_Windows, "Test");
        public static readonly CertLoader ValidLookingTsaCert = new CertLoaderFromRawData(RawData.ValidLookingTsaCert_Cer, RawData.ValidLookingTsaCert_Pfx, "export");
        public static readonly CertLoader TwoEkuTsaCert = new CertLoaderFromRawData(RawData.TwoEkuTsaCert, RawData.TwoEkuTsaPfx, "export");
        public static readonly CertLoader NonCriticalTsaEku = new CertLoaderFromRawData(RawData.NonCriticalTsaEkuCert, RawData.NonCriticalTsaEkuPfx, "export");
        public static readonly CertLoader TlsClientServerCert = new CertLoaderFromRawData(RawData.TlsClientServerEkuCert, RawData.TlsClientServerEkuPfx, "export");
        public static readonly CertLoader NegativeSerialNumber = new CertLoaderFromRawData(RawData.NegativeSerialNumberCert, RawData.NegativeSerialNumberPfx, "1234");

        // Note: the raw data is its own (nested) class to avoid problems with static field initialization ordering.
        private static class RawData
        {
            public static byte[] s_RSAKeyTransfer1Cer =
                 ("308201c830820131a003020102021031d935fb63e8cfab48a0bf7b397b67c0300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657231301e170d3136303431323136323534375a170d31373034313232"
                + "32323534375a301a311830160603550403130f5253414b65795472616e736665723130819f300d06092a864886f70d010101"
                + "050003818d00308189028181009eaab63f5629db5ac0bd74300b43ba61f49189ccc30c001fa96bd3b139f45732cd3c37e422"
                + "ccbb2c598a4c6b3977a516a36ff850a5e914331f7445e86973f5a6cbb590105e933306e240eab6db72d08430cd7316e99481"
                + "a272adef0f2479d0b7c58e89e072364d660fdad1b51a603ff4549a82e8dc914df82bcc6c6c232985450203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d01010505000381810048c83e6f45d73a111c67e8f9f9c2d646292b"
                + "75cec52ef0f9ae3e1639504aa1759512c46527fcf5476897d3fb6fc515ff1646f8f8bc09f84ea6e2ad04242d3fb9b190b816"
                + "86b73d334e8b3afa7fb8eb31483efc0c7ccb0f8c1ca94d8be4f0daade4498501d02e6f92dd7b2f4401550896eb511ef14417"
                + "cbb5a1b360d67998d334").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSAKeyTransfer1Pfx =
                 ("308205d20201033082058e06092a864886f70d010701a082057f0482057b308205773082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e040818fdedadbb31b101020207d0048202806aa390fa9a4cb071a0daf25765ed69efe039896036c0f0edfc03ebe35d2a"
                + "f2f6a5bc9efd907f3b64ae15ac7f61d830e48810aa096ee37fe442b7bfbceeb92e22c25bd5484baf91460be29e06648485db"
                + "7b10ea92d17983c4d22067396c12e4598541ab989d7beb38bf8a0213fd7c9d49ecd46d319bbb58b1423504cd4145e1b33978"
                + "41306c5ace9eab42d408e05101911adc684e63a8c8c9579ce929e48ce2393af1a63c3180c52bd87475e3edb9763dff731ede"
                + "38fc8043dee375001a59e7d6eec5d686d509efee38ef0e7bddcd7ba0477f6f38ff7172ceaeef94ff56ad4b9533241f404d58"
                + "c2b5d54f1ab8250c56b1a70f57b7fffc640b7037408b8f830263befc031ffe7dbc6bef23f02c1e6e2b541be12009bfb11297"
                + "02fc0559e54d264df9b0d046c73ad1b25056231e5d3c4015bdc4f0a9af70ac28b7241233ecc845ce14484779102a45da2560"
                + "c354ec3e01f26d0e0b9a8b650f811d2ffeba95ec1e5cf6be2d060788c1b18ea4ec8f41e46da734c1216044a10a3e171620ed"
                + "79f7e9dd36972c89d91111c68fd60a94d2aa2a3dbbde0383c7c367f77b70a218ddf9fb4ed7abf94c233ffb2797d9ca3802ed"
                + "77868d3ab5651abb90e4de9ea74854b13603859b308689d770a62b5821e5a5650ecb23ca2894ad7901c7e1d2f22ef97e9092"
                + "f0791e886487a59d380d98c0368d3f2f261e0139714b02010e61aa073ee782b1fe5b6f79d070ef1412a13270138330a2e308"
                + "599e1e7829be9f983202ac0dc1c38d38587defe2741903af35227e4f979a68adef86a8459be4a2d74e5de7f94e114a8ea7e4"
                + "0ea2af6b8a93a747377bdd8ddd83c086bb20ca49854efb931ee689b319f984e5377f5a0f20d0a613326d749af00675c6bc06"
                + "0be528ef90ec6a9b2f9b3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082022706092a864886f70d010701a0820218048202"
                + "14308202103082020c060b2a864886f70d010c0a0103a08201e4308201e0060a2a864886f70d01091601a08201d0048201cc"
                + "308201c830820131a003020102021031d935fb63e8cfab48a0bf7b397b67c0300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657231301e170d3136303431323136323534375a170d31373034313232"
                + "32323534375a301a311830160603550403130f5253414b65795472616e736665723130819f300d06092a864886f70d010101"
                + "050003818d00308189028181009eaab63f5629db5ac0bd74300b43ba61f49189ccc30c001fa96bd3b139f45732cd3c37e422"
                + "ccbb2c598a4c6b3977a516a36ff850a5e914331f7445e86973f5a6cbb590105e933306e240eab6db72d08430cd7316e99481"
                + "a272adef0f2479d0b7c58e89e072364d660fdad1b51a603ff4549a82e8dc914df82bcc6c6c232985450203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d01010505000381810048c83e6f45d73a111c67e8f9f9c2d646292b"
                + "75cec52ef0f9ae3e1639504aa1759512c46527fcf5476897d3fb6fc515ff1646f8f8bc09f84ea6e2ad04242d3fb9b190b816"
                + "86b73d334e8b3afa7fb8eb31483efc0c7ccb0f8c1ca94d8be4f0daade4498501d02e6f92dd7b2f4401550896eb511ef14417"
                + "cbb5a1b360d67998d3343115301306092a864886f70d0109153106040401000000303b301f300706052b0e03021a0414c4c0"
                + "4e0c0b0a20e50d58cb5ce565ba7c192d5d3f041479b53fc5f1f1f493a02cf113d563a247462e8726020207d0").HexToByteArray();

            public static byte[] s_RSAKeyTransfer2Cer =
                 ("308201c830820131a00302010202102bce9f9ece39f98044f0cd2faa9a14e7300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657232301e170d3136303332353231323334325a170d31373033323630"
                + "33323334325a301a311830160603550403130f5253414b65795472616e736665723230819f300d06092a864886f70d010101"
                + "050003818d0030818902818100ea5a3834bfb863ae481b696ea7010ba4492557a160a102b3b4d11c120a7128f20b656ebbd2"
                + "4b426f1a6d40be0a55ca1b53ebdca202d258eebb20d5c662819182e64539360461dd3b5dda4085f10250fc5249cf023976b8"
                + "db2bc5f5e628fdb0f26e1b11e83202cbcfc9750efd6bb4511e6211372b60a97adb984779fdae21ce070203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d0101050500038181004dc6f9fd6054ae0361d28d2d781be590fa8f"
                + "5685fedfc947e315db12a4c47e220601e8c810e84a39b05b7a89f87425a06c0202ad48b3f2713109f5815e6b5d61732dac45"
                + "41da152963e700a6f37faf7678f084a9fb4fe88f7b2cbc6cdeb0b9fdcc6a8a16843e7bc281a71dc6eb8bbc4092d299bf7599"
                + "a3492c99c9a3acf41b29").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSAKeyTransfer2Pfx =
                 ("308205d20201033082058e06092a864886f70d010701a082057f0482057b308205773082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e04080338620310d29656020207d0048202804a94d3b1a1bf43efe3726aa9f0abc90c44585d2f0aee0864b4d574cd2cc1"
                + "dca4a353b102779e072ed6072d3c083b83974e74069b353ba8ac8be113228e0225993f5ecb7293ab1a6941bef75f7bcb0e3b"
                + "e6902832be46b976e94c6a0bc6865822ff07371551d206e300558da67cf972d89c3d181beb86d02f5523baa8351b88992654"
                + "a4c507e136dd32120530585a25424fe40f9962b910e08fb55f582c3764946ba7f6d92520decfc9faa2d5e180f9824e5ed4c8"
                + "c57e549a27950e7a875f2ed450035a69de6d95ec7bd9e30b65b8563fdd52809a4a1fc960f75c817c72f98afb000e8a8a33be"
                + "f62e458c2db97b464121489bf3c54de45e05f9c3e06c21892735e3f2d9353a71febcd6a73a0af3c3fc0922ea71bdc483ed7e"
                + "5653740c107cfd5e101e1609c20061f864671ccb45c8b5b5b7b48436797afe19de99b5027faf4cead0fd69d1987bbda5a0a4"
                + "0141495998d368d3a4747fc370205eed9fc28e530d2975ca4084c297a544441cf46c39fb1f0f42c65b99a6c9c970746012ad"
                + "c2be15fbbc803d5243f73fdec50bdee0b74297bd30ca3ea3a1dc623db6a199e93e02053bd1a6ca1a00a5c6090de1fa10cdd5"
                + "b5541bd5f5f92ff60a139c50deff8768e7b242018611efd2cce0d9441f3c8b207906345a985617ba5e98e7883c9b925ba17d"
                + "c4fadddbbe025cecd24bb9b95cae573a8a24ceb635eb9f663e74b0084a88f4e8e0d2baf767be3abe5b873695989a0edac7bd"
                + "092de79c3b6427dcbedee0512918fc3f7a45cd6898701673c9ed9f2f873abb8aa64cec7b8d350e8c780c645e50ce607a1afd"
                + "bcefba6cf5cebbc766d1e61d78fbef7680b38dd0f32133ceb39c6c9cabd0b33af9f7ef73c94854b57cf68e61997b61393a0b"
                + "6fc37f8834157e0c9fba3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082022706092a864886f70d010701a0820218048202"
                + "14308202103082020c060b2a864886f70d010c0a0103a08201e4308201e0060a2a864886f70d01091601a08201d0048201cc"
                + "308201c830820131a00302010202102bce9f9ece39f98044f0cd2faa9a14e7300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657232301e170d3136303332353231323334325a170d31373033323630"
                + "33323334325a301a311830160603550403130f5253414b65795472616e736665723230819f300d06092a864886f70d010101"
                + "050003818d0030818902818100ea5a3834bfb863ae481b696ea7010ba4492557a160a102b3b4d11c120a7128f20b656ebbd2"
                + "4b426f1a6d40be0a55ca1b53ebdca202d258eebb20d5c662819182e64539360461dd3b5dda4085f10250fc5249cf023976b8"
                + "db2bc5f5e628fdb0f26e1b11e83202cbcfc9750efd6bb4511e6211372b60a97adb984779fdae21ce070203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d0101050500038181004dc6f9fd6054ae0361d28d2d781be590fa8f"
                + "5685fedfc947e315db12a4c47e220601e8c810e84a39b05b7a89f87425a06c0202ad48b3f2713109f5815e6b5d61732dac45"
                + "41da152963e700a6f37faf7678f084a9fb4fe88f7b2cbc6cdeb0b9fdcc6a8a16843e7bc281a71dc6eb8bbc4092d299bf7599"
                + "a3492c99c9a3acf41b293115301306092a864886f70d0109153106040401000000303b301f300706052b0e03021a04143cdb"
                + "6a36dfd2288ba4e3771766d7a5289c04419704146c84193dc4f3778f21197d11ff994d8bf4822049020207d0").HexToByteArray();

            public static byte[] s_RSAKeyTransfer3Cer =
                 ("308201c830820131a00302010202104497d870785a23aa4432ed0106ef72a6300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657233301e170d3136303332353231323335355a170d31373033323630"
                + "33323335355a301a311830160603550403130f5253414b65795472616e736665723330819f300d06092a864886f70d010101"
                + "050003818d0030818902818100bbc6fe8702a4e92eadb9b0f41577c0fffc731411c6f87c27c9ef7c2e2113d4269574f44f2e"
                + "90382bd193eb2f57564cf00092172d91a003e7252a544958b30aab6402e6fba7e442e973d1902e383f6bc4a4d8a00e60b3f3"
                + "3a032bdf6bedb56acb0d08669b71dd7b35f5d39d9914f5e111e1cd1559eb741a3075d673c39e7850a50203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d01010505000381810058abccbf69346360351d55817a61a6091b0b"
                + "022607caeb44edb6f05a91f169903608d7391b245ac0dcbe052e16a91ac1f8d9533f19f6793f15cb6681b2cbaa0d8e83d77b"
                + "5207e7c70d843deda8754af8ef1029e0b68c35d88c30d7da2f85d1a20dd4099facf373341b50a8a213f735421062e1477459"
                + "6e27a32e23b3f3fcfec3").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSAKeyTransfer3Pfx =
                 ("308205d20201033082058e06092a864886f70d010701a082057f0482057b308205773082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e0408a9197ad512c316b5020207d004820280b1c213fa87f3906cde3502249830a01d1d636d0058bd8d6172222544c35a"
                + "9676f390a5ef1d52f13fae2f04fe2ca1bcb9914296f97fdf729a52e0c3472c9f7ae72bd746f0a66b0c9363fae0328ad063fa"
                + "45d35cc2679c85e970c7420ad036012ce553ef47ed8fe594917739aab1123be435a0ca88ac4b85cf3d341d4aeb2c6816d8fc"
                + "a2e9611224b42f0ca00bde4f25db460200f25fe99ed4fd0236e4d00c48085aec4734f0bce7e6c8fea08b11a2a7214f4a18c0"
                + "fa4b732c8dae5c5857f2edec27fa94eb17ac05d1d05b321b01c1368231ff89c46c6378abf67cb751156370bbcc35591e0028"
                + "d4ace5158048d9d25b00e028b7766f1c74ade9603a211aad241fc3b7599a2b15f86846dfdc106f49cf56491b3f6ff451d641"
                + "400f38fabcdb74a4423828b041901fa5d8c528ebf1cc6169b08eb14b2d457acb6970a11ccaa8fbc3b37b6454803b07b1916e"
                + "2ad3533f2b72721625c11f39a457033744fde3745c3d107a3f1e14118e04db41ca8970a383e8706bcf8ba5439a4cb360b250"
                + "4fcae3dbfb54af0154f9b813ad552f2bdbc2a9eb61d38ae5e6917990cbeb1c5292845637c5fed477dabbed4198a2978640ba"
                + "7db22c85322115fa9027ad418a61e2e31263da3776398faaaab818aae6423c873bd393f558fa2fc05115b4983d35ecfeae13"
                + "601519a53c7a77b5688aeddc6f210a65303eeb0dbd7e3a5ec94d7552cf4cbe7acebf5e4e10abaccd2e990f1cf217b98ad9b5"
                + "06820f7769a7c5e61d95462918681c2b111faf29f13e3615c4c5e75426dbcd903c483590434e8ab1965dc620e7d8bebea36f"
                + "53f1bc0807933b0ef9d8cc1b36b96aff8288e9a8d1bba24af562dfeb497b9a58083b71d76dacd6f2ce67cb2593c6f06472ef"
                + "e508012c34f40d87e0be3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082022706092a864886f70d010701a0820218048202"
                + "14308202103082020c060b2a864886f70d010c0a0103a08201e4308201e0060a2a864886f70d01091601a08201d0048201cc"
                + "308201c830820131a00302010202104497d870785a23aa4432ed0106ef72a6300d06092a864886f70d0101050500301a3118"
                + "30160603550403130f5253414b65795472616e7366657233301e170d3136303332353231323335355a170d31373033323630"
                + "33323335355a301a311830160603550403130f5253414b65795472616e736665723330819f300d06092a864886f70d010101"
                + "050003818d0030818902818100bbc6fe8702a4e92eadb9b0f41577c0fffc731411c6f87c27c9ef7c2e2113d4269574f44f2e"
                + "90382bd193eb2f57564cf00092172d91a003e7252a544958b30aab6402e6fba7e442e973d1902e383f6bc4a4d8a00e60b3f3"
                + "3a032bdf6bedb56acb0d08669b71dd7b35f5d39d9914f5e111e1cd1559eb741a3075d673c39e7850a50203010001a30f300d"
                + "300b0603551d0f040403020520300d06092a864886f70d01010505000381810058abccbf69346360351d55817a61a6091b0b"
                + "022607caeb44edb6f05a91f169903608d7391b245ac0dcbe052e16a91ac1f8d9533f19f6793f15cb6681b2cbaa0d8e83d77b"
                + "5207e7c70d843deda8754af8ef1029e0b68c35d88c30d7da2f85d1a20dd4099facf373341b50a8a213f735421062e1477459"
                + "6e27a32e23b3f3fcfec33115301306092a864886f70d0109153106040401000000303b301f300706052b0e03021a0414cd11"
                + "0833d653f2e18d2afb2de74689ff0446ec7d0414f2ca1c390db19317697044b9012ef6864e0f05cc020207d0").HexToByteArray();

            public static byte[] s_RSAKeyTransferCapi1Cer =
                 ("3082020c30820179a00302010202105d2ffff863babc9b4d3c80ab178a4cca300906052b0e03021d0500301e311c301a0603"
                + "55040313135253414b65795472616e736665724361706931301e170d3135303431353037303030305a170d32353034313530"
                + "37303030305a301e311c301a060355040313135253414b65795472616e73666572436170693130819f300d06092a864886f7"
                + "0d010101050003818d0030818902818100aa272700586c0cc41b05c65c7d846f5a2bc27b03e301c37d9bff6d75b6eb6671ba"
                + "9596c5c63ba2b1af5c318d9ca39e7400d10c238ac72630579211b86570d1a1d44ec86aa8f6c9d2b4e283ea3535923f398a31"
                + "2a23eaeacd8d34faaca965cd910b37da4093ef76c13b337c1afab7d1d07e317b41a336baa4111299f99424408d0203010001"
                + "a3533051304f0603551d0104483046801015432db116b35d07e4ba89edb2469d7aa120301e311c301a060355040313135253"
                + "414b65795472616e73666572436170693182105d2ffff863babc9b4d3c80ab178a4cca300906052b0e03021d050003818100"
                + "81e5535d8eceef265acbc82f6c5f8bc9d84319265f3ccf23369fa533c8dc1938952c5931662d9ecd8b1e7b81749e48468167"
                + "e2fce3d019fa70d54646975b6dc2a3ba72d5a5274c1866da6d7a5df47938e034a075d11957d653b5c78e5291e4401045576f"
                + "6d4eda81bef3c369af56121e49a083c8d1adb09f291822e99a429646").HexToByteArray();

            // Password = "1111"
            //
            // Built by:
            //
            //   makecert -r -len 1024 -n "CN=RSAKeyTransferCapi1" -b 04/15/2015 -e 04/15/2025 RSAKeyTransferCapi1.cer -sv RSAKeyTransferCapi1.pvk -sky exchange
            //   pvk2pfx.exe -pvk RSAKeyTransferCapi1.pvk -spc RSAKeyTransferCapi1.cer -pfx RSAKeyTransferCapi1.pfx -po 1111
            //  
            public static byte[] s_RSAKeyTransferCapi1Pfx =
                 ("30820626020103308205e206092a864886f70d010701a08205d3048205cf308205cb3082035806092a864886f70d010701a0"
                + "82034904820345308203413082033d060b2a864886f70d010c0a0102a08202b6308202b2301c060a2a864886f70d010c0103"
                + "300e0408dbd82a9abd7c1a2b020207d004820290768873985e74c2ece506531d348d8b43f2ae8524a2bcc737eeb778fac1ee"
                + "b21f82deb7cf1ba54bc9a865be8294de23e6648ffb881ae2f0132265c6dacd60ae55df1497abc3eb9181f47cb126261ea66f"
                + "d22107bbcdb8825251c60c5179ef873cb7e047782a4a255e3e9d2e0dd33f04cde92f9d268e8e4daf8ba74e54d8b279a0e811"
                + "9a3d0152608c51331bbdd23ff65da492f85809e1d7f37af9ae00dca796030a19e517e7fe2572d4502d4738fd5394ee369216"
                + "fb64cf84beab33860855e23204156dcf774fac18588f1c1ca1a576f276e9bfbf249449842f193020940a35f163378a2ce7da"
                + "37352d5b0c7c3ac5eb5f21ed1921a0076523b2e66a101655bb78d4ecc22472ac0151b7e8051633747d50377258ab19dcb22e"
                + "e09820876607d3291b55bba73d713d6689486b310507316b4f227383e4869628ad31f0b431145d45f4f38f325772c866a20e"
                + "0b442088cbf663e92e8ee82dd495fba8d40345474a384bb3b80b49ca1d66eef5321235135dcc0a5425e4bf3b8ce5c2469e2a"
                + "c0f8d53aab276361d9a2ff5c974c6e6b66126158676331fe7f74643fd1e215b22d7799846651350ed0f1f21a67ac6b3bfd62"
                + "7defb235ef8732d772d1c4bea2ae80c165f0182f547ea7a3f3366288f74c030689988a9838c27b10a48737a620d8220f68b4"
                + "ea8d8eb26298d5359d54a59c6be6716cefc12c929e17bb71c57c560659a7757ba8ac08ae90794474e50f0e87a22e2b7c3ebd"
                + "061390928bf48c6c6200c225f7025eab20f5f6fee5dc41682b2d4a607c8c81964b7d52651e5a62a41f4e8ea3982c294a4aee"
                + "8a67dc36a8b34b29509a4868c259dc205d1e8a3b6259a76a147f002f3bfbc8378e8edd230a34f9cd5f13ce6651b10394709d"
                + "5092bb6a70d8c2816f1c0e44cd45dfa7c2d94aa32112d79cb44a3174301306092a864886f70d010915310604040100000030"
                + "5d06092b060104018237110131501e4e004d006900630072006f0073006f006600740020005300740072006f006e00670020"
                + "00430072007900700074006f0067007200610070006800690063002000500072006f007600690064006500723082026b0609"
                + "2a864886f70d010701a082025c048202583082025430820250060b2a864886f70d010c0a0103a082022830820224060a2a86"
                + "4886f70d01091601a0820214048202103082020c30820179a00302010202105d2ffff863babc9b4d3c80ab178a4cca300906"
                + "052b0e03021d0500301e311c301a060355040313135253414b65795472616e736665724361706931301e170d313530343135"
                + "3037303030305a170d3235303431353037303030305a301e311c301a060355040313135253414b65795472616e7366657243"
                + "6170693130819f300d06092a864886f70d010101050003818d0030818902818100aa272700586c0cc41b05c65c7d846f5a2b"
                + "c27b03e301c37d9bff6d75b6eb6671ba9596c5c63ba2b1af5c318d9ca39e7400d10c238ac72630579211b86570d1a1d44ec8"
                + "6aa8f6c9d2b4e283ea3535923f398a312a23eaeacd8d34faaca965cd910b37da4093ef76c13b337c1afab7d1d07e317b41a3"
                + "36baa4111299f99424408d0203010001a3533051304f0603551d0104483046801015432db116b35d07e4ba89edb2469d7aa1"
                + "20301e311c301a060355040313135253414b65795472616e73666572436170693182105d2ffff863babc9b4d3c80ab178a4c"
                + "ca300906052b0e03021d05000381810081e5535d8eceef265acbc82f6c5f8bc9d84319265f3ccf23369fa533c8dc1938952c"
                + "5931662d9ecd8b1e7b81749e48468167e2fce3d019fa70d54646975b6dc2a3ba72d5a5274c1866da6d7a5df47938e034a075"
                + "d11957d653b5c78e5291e4401045576f6d4eda81bef3c369af56121e49a083c8d1adb09f291822e99a429646311530130609"
                + "2a864886f70d0109153106040401000000303b301f300706052b0e03021a041463c18f4fec17cf06262e8acd744e18b8ab7b"
                + "8f280414134ec4a25653b142c3d3f9999830f2ac66ef513b020207d0").HexToByteArray();

            public static byte[] s_RSASha256KeyTransfer1Cer =
                 ("308201d43082013da003020102021072c6c7734916468c4d608253da017676300d06092a864886f70d01010b05003020311e"
                + "301c060355040313155253415368613235364b65795472616e7366657231301e170d3136303431383130353934365a170d31"
                + "37303431383136353934365a3020311e301c060355040313155253415368613235364b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100cad046de3a7f6dc78fc5a4e01d1f7d90db596f586334d5708a"
                + "ecb8e52d6bb912c0b5ec9633a82b4abac4c2860c766f2fdf1c905c4a72a54adfd041adabe5f2afd1e2ad88615970e818dc3d"
                + "4d00bb6c4ce94c5eb4e3efedd80d14c3d295ea471ae430cbb20b071582f1396369fbe90c14aa5f85b8e3b14011d81fbd41ec"
                + "b1495d0203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010b050003818100baed2a5ae2d1"
                + "1ee4209c0694c790e72e3e8ad310b2506b277d7c001b09f660d48dba846ac5bbef97653613adf53d7624fc9b2b337f25cb33"
                + "74227900cfefbe2fdac92b4f769cf2bf3befb485f282a85bfb09454b797ce5286de560c219fb0dd6fce0442adbfef4f767e9"
                + "ac81cf3e9701baf81efc73a0ed88576adff12413b827").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSASha256KeyTransfer1Pfx =
                 ("308205de0201033082059a06092a864886f70d010701a082058b04820587308205833082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e040829e4911057aa5fb6020207d00482028052e016e1e339ca6a648ab1e152813899bd2ec0de1e34804c33e109cf2136"
                + "d42edc0d5ff8a005939ec38d4284aa0cfda295e801b701855c3c129e9311dc80b3538ba76d3164d48d83a73949d695f42294"
                + "75469f262c807767bc5c12bb83b2c4857fa9f8c7c519143136ba93ab93e17ad4b0b63cf6449708e6128425b00eaeae6bc5b6"
                + "7ff092673c9aabbbb63e90424295f0ae828bcd00f5ad85fe8384711ca5fffd4cbfe57ddbc3e5bb1df19e6fd7640fbd8d4516"
                + "f8d2d5ec84baca72ac42b50e77be0055dfdbbbe9c6de42c06fc86de8fbfc6231db89b30065d534e76aa851833b6c9c651288"
                + "c12f87ba12ae429e9bec0b22297c666046355ebd5a54dc7f13a55e0ebd53c768f69eee57d6041263f5bdf1c4c5b2b55dfb9b"
                + "38171aaed0d21fd5a41e0ef760db42f373c9007e1df47fd79ba9b41528c9c02dffdd04472265763ae94f4e05b86976a2c459"
                + "093d8e6bb0d0c5da5994fe3edbdf843b67e8e4c4daf59351788bf8b96da116aecbb95d52bf727ff10ca41340112f0bcb41e0"
                + "b8373a6e55727c745b77cf1944b74fa447ed0a6d93b8e43fd6e4b4b3e0d49d03ee2ee12d15519406c49a4c1be70de5171c93"
                + "d056e9f47b8a96d50f01873be4c596590f1247a2f2822dea9339fa87dd49545b559e0225ab738ecc0b054155749670d412be"
                + "472d13dfb0a8c8f56b3c0be1aa0d9195ba937b0c2119c702a0be1f83e1b4a77375ed1654e3dcf6b8ce119db3ac7cd440369a"
                + "b0b964e0b526b865680015cc3046a20badeaca4543ce65042ff5eb691e93232754a7b34fd8b6833c2625fdfdc59d80b3dcb4"
                + "ce70d1833ecf6344bb7331e46b71bb1592b6d814370548ee2b2f4df207696be87d2e1e0c5dc0ca528e5a231802cbb7853968"
                + "beb6ceb1b3a2998ecd313174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082023306092a864886f70d010701a0820224048202"
                + "203082021c30820218060b2a864886f70d010c0a0103a08201f0308201ec060a2a864886f70d01091601a08201dc048201d8"
                + "308201d43082013da003020102021072c6c7734916468c4d608253da017676300d06092a864886f70d01010b05003020311e"
                + "301c060355040313155253415368613235364b65795472616e7366657231301e170d3136303431383130353934365a170d31"
                + "37303431383136353934365a3020311e301c060355040313155253415368613235364b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100cad046de3a7f6dc78fc5a4e01d1f7d90db596f586334d5708a"
                + "ecb8e52d6bb912c0b5ec9633a82b4abac4c2860c766f2fdf1c905c4a72a54adfd041adabe5f2afd1e2ad88615970e818dc3d"
                + "4d00bb6c4ce94c5eb4e3efedd80d14c3d295ea471ae430cbb20b071582f1396369fbe90c14aa5f85b8e3b14011d81fbd41ec"
                + "b1495d0203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010b050003818100baed2a5ae2d1"
                + "1ee4209c0694c790e72e3e8ad310b2506b277d7c001b09f660d48dba846ac5bbef97653613adf53d7624fc9b2b337f25cb33"
                + "74227900cfefbe2fdac92b4f769cf2bf3befb485f282a85bfb09454b797ce5286de560c219fb0dd6fce0442adbfef4f767e9"
                + "ac81cf3e9701baf81efc73a0ed88576adff12413b8273115301306092a864886f70d0109153106040401000000303b301f30"
                + "0706052b0e03021a0414282ee1780ac2a08b2783b1f8f7c855fb1a53ce9e04143fad59471323dc979f3bf29b927e54eca677"
                + "7576020207d0").HexToByteArray();

            public static byte[] s_RSASha384KeyTransfer1Cer =
                 ("308201d43082013da00302010202103c724fb7a0159a9345caac9e3df5f136300d06092a864886f70d01010c05003020311e"
                + "301c060355040313155253415368613338344b65795472616e7366657231301e170d3136303431383131303530365a170d31"
                + "37303431383137303530365a3020311e301c060355040313155253415368613338344b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100e6b46b0e6f4f6df724081e11f201b9fbb07f2b6db2b868f607"
                + "68e2b5b843f690ca5e8d48f439d8b181ace2fb27dfa07eff0324642d6c9129e2d95e136702f6c31fe3ccf3aa87ba9f1b6f7b"
                + "acd07156ff3dd2a7f4c70356fb94b0adbde6819383c19bbefb4a6d1d6491a770d5f9feb11bcb3e5ac99cb153984dee0910e4"
                + "b57f8f0203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010c0500038181003842cc95a680"
                + "c8a31534a461d061a4706a0aba52b7a1c709c2f1e3b94acf6dc0930b74e63e3babf3c5b11c8f8a888722d9f23c7e0a8c9b09"
                + "90ebcdbce563b8d4209efc1b04750f46c8c6117ccb96b26b5f02b0b5f961ab01b0c3b4cdb2530cbc5dcf37786712a3476ce7"
                + "32c5c544c328db5ebc3a338b18fe32aedaffedd973ef").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSASha384KeyTransfer1Pfx =
                 ("308205de0201033082059a06092a864886f70d010701a082058b04820587308205833082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e040856d7d59810ce8b17020207d00482028082012797edb5f74429bb6b91dd1e24aa32a19b89d92fd486e826773a7a11"
                + "03a9b49d98c6b7e97d411d19b44cd79559964f31cb6f0443c70d687c390d31c656ee3887391ae1735c142d891ec8337c5dc4"
                + "d6b5a4f09400a4cc35dd8dbde831f7625b7afedf4990294988b0b32b2889c97cd85c2568ffef332be83232449dd4083a43d4"
                + "89e654520eb922239379b5e9f5dfc1e64972339dee27dfdd874e2ee2b85f941f3b313ab881571c3a5a9b292d8c82d79d74a0"
                + "2d78dd5cfce366b3a914b61b861b35948757d137e5d53589a0fa2f1b4d06ee6b4aa4b8d3f526b059637b236ceb2de128d7bd"
                + "f91c12612d09e1cb4bed1b5e336fb56424b68dcc6d6cd5d90f666047c8b181526a60622027d322db0172046c23e84a3c725e"
                + "45ce774df037cafb74b359c3ec6874dce98673d9f7581f54dcb6e3c40583de2de6aaf6739bba878362e9bfab331cab2eb22d"
                + "3b130dec4eedf55a7ed8d5960e9f037209f9c1ef584c6dd5de17245d0da62c54420dc862b6648418d2aa9797f86a2cd0ecf6"
                + "abcbeb16907d8f44021690682a4e1286cd3f9aea4866108b3c968cf4b80a39c60436079617346861662e01a5419d8cebe2c6"
                + "e186141e42baf7cfc596270dbab8db03da9bd501daa426e24aa2d8ccf4d4512a8dce3ae8954be69b5c3a70fac587ac91ad97"
                + "fb427c8118659b710b57183c4fd16ffd276834e2fe45d74e175f3f5077783cdd7668b4e87217512ceb7f3e64715ba22bbab7"
                + "0d1b3485820c16304758cf1dd0b806d801f1185bb14d12f2c147ec65b95088077dec23498ebe40a952727c559c7af5cf20f1"
                + "f491f4123db093dc1a67014c3db46c11c7d5833b15167c91138eba6b4badf869aefba5fbea523a5ad02bb676db6039e7aabd"
                + "44f0702d59cf3d1ad9bb3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082023306092a864886f70d010701a0820224048202"
                + "203082021c30820218060b2a864886f70d010c0a0103a08201f0308201ec060a2a864886f70d01091601a08201dc048201d8"
                + "308201d43082013da00302010202103c724fb7a0159a9345caac9e3df5f136300d06092a864886f70d01010c05003020311e"
                + "301c060355040313155253415368613338344b65795472616e7366657231301e170d3136303431383131303530365a170d31"
                + "37303431383137303530365a3020311e301c060355040313155253415368613338344b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100e6b46b0e6f4f6df724081e11f201b9fbb07f2b6db2b868f607"
                + "68e2b5b843f690ca5e8d48f439d8b181ace2fb27dfa07eff0324642d6c9129e2d95e136702f6c31fe3ccf3aa87ba9f1b6f7b"
                + "acd07156ff3dd2a7f4c70356fb94b0adbde6819383c19bbefb4a6d1d6491a770d5f9feb11bcb3e5ac99cb153984dee0910e4"
                + "b57f8f0203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010c0500038181003842cc95a680"
                + "c8a31534a461d061a4706a0aba52b7a1c709c2f1e3b94acf6dc0930b74e63e3babf3c5b11c8f8a888722d9f23c7e0a8c9b09"
                + "90ebcdbce563b8d4209efc1b04750f46c8c6117ccb96b26b5f02b0b5f961ab01b0c3b4cdb2530cbc5dcf37786712a3476ce7"
                + "32c5c544c328db5ebc3a338b18fe32aedaffedd973ef3115301306092a864886f70d0109153106040401000000303b301f30"
                + "0706052b0e03021a041429bd86de50f91b8f804b2097b1d9167ca56577f40414b8714b8172fa1baa384bed57e3ddb6d1851a"
                + "f5e9020207d0").HexToByteArray();

            public static byte[] s_RSASha512KeyTransfer1Cer =
                 ("308201d43082013da00302010202102f5d9d58a5f41b844650aa233e68f105300d06092a864886f70d01010d05003020311e"
                + "301c060355040313155253415368613531324b65795472616e7366657231301e170d3136303431383131303532355a170d31"
                + "37303431383137303532355a3020311e301c060355040313155253415368613531324b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100b2eca20240da8486b1a933ade62ad8781ef30d4434ebbc9b3f"
                + "c9c550d0f9a75f4345b5520f3d0bafa63b8037785d1e8cbd3efe9a22513dc8b82bcd1d44bf26bd2c292205ca3e793ff1cb09"
                + "e0df4afefb542362bc148ea2b76053d06754b4a37a535afe63b048282f8fb6bd8cf5dc5b47b7502760587f84d9995acbf1f3"
                + "4a3ca10203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010d050003818100493d857684d2"
                + "7468dd09926d20933254c7c79645f7b466e7b4a90a583cedba1c3b3dbf4ccf1c2506eb392dcf15f53f964f3c3b519132a38e"
                + "b966d3ea397fe25457b8a703fb43ddab1c52272d6a12476df1df1826c90fb679cebc4c04efc764fd8ce3277305c3bcdf1637"
                + "91784d778663194097180584e5e8ab69039908bf6f86").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSASha512KeyTransfer1Pfx =
                 ("308205de0201033082059a06092a864886f70d010701a082058b04820587308205833082034806092a864886f70d010701a0"
                + "82033904820335308203313082032d060b2a864886f70d010c0a0102a08202a6308202a2301c060a2a864886f70d010c0103"
                + "300e04083a0e344b65dd4e27020207d00482028014464df9f07d2cb37a28607570130de5877e829e759040976866afc831db"
                + "4d2741734ae53ea5eb80c1080dae7b0a2acddabd3d47b1ed5f3051455429308f3b7b0b48c5a4dbc5d718534472c746ce62f1"
                + "bbb8c5d178c1d3e91efdbd4f56569517bcadf3c81dbe4c34746194e47bcf46b74cd1880d7bd12d9b819b462fbcf6f51f3972"
                + "2858c9b9af8975bfefd7f007928b39e11d50b612761d03e566b992f92e9c9873d138c937fc43fe971db4c8e57b51aeef4ed0"
                + "022ec76c3bb4bd9f2395b99585449303a6d68183edf6e5dda1885531bee10b7cf6509390f4ee6a37ed2931d658548bd6390f"
                + "a7094fdf017166309074c00581d2b7dcaaee657f9c48e08edf636004dc5e60486dd022c45058700fe682472b371380948792"
                + "74c2a20dd9e07e149e7ab52157db748160ad81f91019297baa58ce68656b0b2f7c9ac88b3da6920c2a5eab7bcc2629974f8a"
                + "6c8bf33629af05e4e34d5d24393448e9751b7708f5915b0fd97a5af4dd5a37d71b18b6526316cbc65b1c6af8a6779acbc470"
                + "2381f027bdb118cb84e9005b02a8bd2d02365d280cffb04831f877de7bd3d3287f11beed8978a5389e2b28317eb90569781f"
                + "94f66f672736a09b4a7caeaaefd1909f2d20255df51512dbd08ec6125455d932b626bdfd3c4f669148fa783671f90b59ceff"
                + "560c338f92cbe8bf7fbab4db3e9b943effac747eb34f06bd72aee961ed31742caa2a9934a5fe4685677ecbca6fb1b1c0b642"
                + "b4f71d55d0e2cb1dc10ce845514090cc117a875c4d10c0ce367e31091144eacd7e600792d61d036bde020e3bb9a004a7dd1a"
                + "cf03541b6fff3bcef4c30df05d98b75688320685261b2b34813407b20a7c92a04eeb46cb7e618a6ee32154728ba6735668f4"
                + "11abece4ba07426a394b3174301306092a864886f70d0109153106040401000000305d06092b060104018237110131501e4e"
                + "004d006900630072006f0073006f0066007400200053006f0066007400770061007200650020004b00650079002000530074"
                + "006f0072006100670065002000500072006f007600690064006500723082023306092a864886f70d010701a0820224048202"
                + "203082021c30820218060b2a864886f70d010c0a0103a08201f0308201ec060a2a864886f70d01091601a08201dc048201d8"
                + "308201d43082013da00302010202102f5d9d58a5f41b844650aa233e68f105300d06092a864886f70d01010d05003020311e"
                + "301c060355040313155253415368613531324b65795472616e7366657231301e170d3136303431383131303532355a170d31"
                + "37303431383137303532355a3020311e301c060355040313155253415368613531324b65795472616e736665723130819f30"
                + "0d06092a864886f70d010101050003818d0030818902818100b2eca20240da8486b1a933ade62ad8781ef30d4434ebbc9b3f"
                + "c9c550d0f9a75f4345b5520f3d0bafa63b8037785d1e8cbd3efe9a22513dc8b82bcd1d44bf26bd2c292205ca3e793ff1cb09"
                + "e0df4afefb542362bc148ea2b76053d06754b4a37a535afe63b048282f8fb6bd8cf5dc5b47b7502760587f84d9995acbf1f3"
                + "4a3ca10203010001a30f300d300b0603551d0f040403020520300d06092a864886f70d01010d050003818100493d857684d2"
                + "7468dd09926d20933254c7c79645f7b466e7b4a90a583cedba1c3b3dbf4ccf1c2506eb392dcf15f53f964f3c3b519132a38e"
                + "b966d3ea397fe25457b8a703fb43ddab1c52272d6a12476df1df1826c90fb679cebc4c04efc764fd8ce3277305c3bcdf1637"
                + "91784d778663194097180584e5e8ab69039908bf6f863115301306092a864886f70d0109153106040401000000303b301f30"
                + "0706052b0e03021a041401844058f6e177051a87eedcc55cc4fa8d567ff10414669cb82c9cc3ceb4d3ca9f65bd57ba829616"
                + "60d9020207d0").HexToByteArray();

            public static byte[] s_DHKeyAgree1Cer =
                 ("3082041930820305a00302010202100ae59b0cb8119f8942eda74163413a02300906052b0e03021d0500304f314d304b0603"
                + "5504031e44004d0061006e006100670065006400200050004b00430053002300370020005400650073007400200052006f00"
                + "6f007400200041007500740068006f0072006900740079301e170d3136303431333132313630315a170d3339313233313233"
                + "353935395a301f311d301b06035504031314446648656c6c654b657941677265656d656e7431308201b63082012b06072a86"
                + "48ce3e02013082011e02818100b2f221e2b4649401f817557771e4f2ca1c1309caab3fa4d85b03dc1ea13c8395665eb4d05a"
                + "212b33e1d727403fec46d30ef3c3fd58cd5b621d7d30912f2360676f16b206aa419dba39b95267b42f14f6500b1729de2d94"
                + "ef182ed0f3042fd3850a7398808c48f3501fca0e929cec7a9594e98bccb093c21ca9b7dbdfcdd733110281805e0bed02dd17"
                + "342f9f96d186d2cc9e6ff57f5345b44bfeeb0da936b37bca62e2e508d9635a216616abe777c3fa64021728e7aa42cfdae521"
                + "01c6a390c3eb618226d8060ceacdbc59fa43330ad41e34a604b1c740959b534f00bd6cf0f35b62d1f8de68d8f37389cd435d"
                + "764b4abec5fc39a1e936cdf52a8b73e0f4f37dda536902150093ced62909a4ac3aeca9982f68d1eed34bf055b30381840002"
                + "81804f7e72a0e0ed4aae8e498131b0f23425537b9a28b15810a3c1ff6f1439647f4e55dcf73e72a7573ce609a5fb5c5dc3dc"
                + "daa883b334780c232ea12b3af2f88226775db48f4b800c9ab1b54e7a26c4c0697bbd5e09355e3b4ac8005a89c65027e1d0d7"
                + "091b6aec8ede5dc72e9bb0d3597915d50da58221673ad8a74e76b2a79f25a38194308191300c0603551d130101ff04023000"
                + "3081800603551d010479307780109713ac709a6e2cc6aa54b098e5557cd8a151304f314d304b06035504031e44004d006100"
                + "6e006100670065006400200050004b00430053002300370020005400650073007400200052006f006f007400200041007500"
                + "740068006f00720069007400798210d581eafe596cd7a34d453011f4a4b6f0300906052b0e03021d05000382010100357fbe"
                + "079401e111bf80db152752766983c756eca044610f8baab67427dc9b5f37df736da806e91a562939cf876a0998c1232f31b9"
                + "9cf38f0e34d39c7e8a2cc04ed897bfdc91f7f292426063ec3ec5490e35c52a7f98ba86a4114976c45881373dacc95ad3e684"
                + "7e1e28bb58e4f7cfc7138a56ce75f01a8050194159e1878bd90f9f580f63c6dd41e2d15cd80dc0a8db61101df9009d891ec2"
                + "28f70f3a0a37358e7917fc94dfeb6e7cb176e8f5dbfa1ace2af6c0a4306e22eb3051e7705306152ce87328b24f7f153d565b"
                + "73aef677d25ae8657f81ca1cd5dd50404b70b9373eadcd2d276e263105c00607a86f0c10ab26d1aafd986313a36c70389a4d"
                + "1a8e88").HexToByteArray();

            public static byte[] s_RSAKeyTransferCer_ExplicitSki =
                ("3082033E30820226A003020102020900B5EFA7E1E80518B4300D06092A864886F70D01010B0500304D310B3009060355"
                + "04061302515A310D300B060355040813044C616E643111300F060355040713084D7974686963616C311C301A06035504"
                + "03131353656C662D5369676E6564204578616D706C65301E170D3136303632383030323034355A170D31363037323830"
                + "30323034355A304D310B300906035504061302515A310D300B060355040813044C616E643111300F060355040713084D"
                + "7974686963616C311C301A0603550403131353656C662D5369676E6564204578616D706C6530820122300D06092A8648"
                + "86F70D01010105000382010F003082010A0282010100D95D63618741AD85354BA58242835CD69D7BC2A41173221E899E"
                + "109F1354A87F5DC99EF898881293880D55F86E779E353C226CEA0D1FFCC2EE7227216DDC9116B7DF290A81EC9434CDA4"
                + "408B7C06517B3AFF2C9D0FD458F9FCCDE414849C421402B39D97E197CA0C4F874D65A86EAD20E3041A0701F6ABA063AC"
                + "B418186F9BF657C604776A6358C0F031608673278EFD702A07EE50B6DC1E090EEE5BB873284E6547F612017A26DEC5C2"
                + "7533558F10C1894E899E9F8676D8C0E547B6B5C6EEEF23D06AA4A1532144CF104EB199C324F8E7998DB63B251C7E35A0"
                + "4B7B5AFFD652F5AD228B099863C668772BEEEFF4F60EA753C8F5D0780AAED4CFA7860F1D3D490203010001A321301F30"
                + "1D0603551D0E0416041401952851C55DB594B0C6167F5863C5B6B67AEFE6300D06092A864886F70D01010B0500038201"
                + "0100BB7DDDC4EEC4E31163B84C5C3F4EA1B9314B75D4CCCA8F5339503DC6318E279E73826AD77AC6EA4344F2905D8792"
                + "83D867A66319E1EA383B70CE997A98D545FE258E38C577554E3BBDF42349B98BED9C94B7CE55D51F740EF60ACC75D017"
                + "F3D0D1438F8F86B2453F2308279521EE4AC09046498F60ECEC8867E7BF011C882A514BF85F6C915A70E0383AB5320034"
                + "19A107A9FFEDBF34681AEEE57DF6A3DB3759D605A9269FB694D8EA56A90740529159D725BFD70C9141A38B98D4E88CDC"
                + "31124ABBB4C3D3D49C220CCB6F2F94176B8225A0E2ADDB0F4A72E6B021601CD297AC45A0CAB95EBAC4001C8167899868"
                + "3188DB9364AAD52D4E28169CC898B621FF84").HexToByteArray();

            // password = "1111"
            public static byte[] s_RSAKeyTransferPfx_ExplicitSki =
                ("308209810201033082094706092A864886F70D010701A08209380482093430820930308203E706092A864886F70D0107"
                + "06A08203D8308203D4020100308203CD06092A864886F70D010701301C060A2A864886F70D010C0106300E0408101C5A"
                + "3E2DBE2A9102020800808203A0F58476F5E4741F8834F50ED49D1A3A5B2FC8C345B54255C30556B1426C1BA1D9EE4440"
                + "CD63CD48557B7BDC55D877D656183E2815DEDE92236E036E0D7FD93022174EFA179A85EF76DEE10950EE3BEB004FB118"
                + "C58D4372A319575DB129F5912B385E63E1E83DC420A8FC8C23A283977480281EDDD745D97EC768328875D19FE414D7D9"
                + "9D3B0AAA2FBA77346F82E4E1357C54E142B2F5E929CBD6057801F49ED08A8BD2456918CCEDAD6DAD9A7281C4EFD2FCF5"
                + "6F04EDC5E62E79741024AF8BE401141AA9A9CE08F5D51D4636D8B25F9B3C59B4BC2DD7E60FBABA0A7E8FE15EAECB7221"
                + "3BC22D8CE56987427B41A79333FB4B9BC5DB6E79C2AE4E954F975C343D155C5587BD7206414B9C0529D00C6DB1470C33"
                + "A51D2A9BBDE5CC2352C61B0FB9945487FDB0E26981426BE7CCF44CF494E695D4760060468B7D23BA3C6F9B1762AC4B3A"
                + "428B23A36F275F3FDFD7BAB236197C7C7FB6466A11B05DB39F947FB19EFE9BFED2B18308E2BBD0AB00AA399508194CB2"
                + "1073B1B278BE389A8AA843B610B439AFA056A0EC81EBDF4061D2AB87C9CB840C3E6B92BB2FC30815D5744593862CC34A"
                + "EF1C4B7BBCF640CBA2D1E15E13D3B571FD3C403BC927812B291E53EAE6721C994E343148C10A16053AE560A55DFA5695"
                + "33CA35D83D81643CC7788E7F94C6592F99C09AFB770E9FE1380A1212A646A936BE531BF85F89D19EF57C180E8E3F1F4F"
                + "BD266032095862E3A0F8394E93CEFF2B8ADAD374DFCB8A041DB498618D1D71765EFD1CD5B024AC13B9FF0F96F975B443"
                + "08C14AC60965082CC409AE43D033512CF1B83458D475D2E06A49131894F1D4BFAF5FC4CBADA8566B6312E8DA31D8A397"
                + "273BE77B8395F4CAB4428B22DFE18FD4365C134B7724220D2DCE31938EFCF8E4DFC321E02CF15476BF5EB675F2055205"
                + "9662166A4549904BC6A5E4B8353C43DAC225317F4B4FA963C900F0B0D0E7FC854BE91A1CFF330FE77B03776EABD0326B"
                + "0FB37AC5176CF82530960F423B13299E037285C9324E0A872414ECF35735F58463506EBFB2CC91D790FC0D67E2714287"
                + "960C68FB43A7EE42A14F5397F07E055E75EE4F7D047634907702EEC8ABB08D82C82CEBE2B042B1F20367DFDB839B82AF"
                + "88F72272AE91DA94CD9B334343196889381FE307A76BE0B627EE32D827582A7CD68BF467D954805030753FA8DABFCC21"
                + "E68A77E2A76F9E95E61A2FBCA1C8FFC2CE272E9D125E5C65759529BF3FDD2E28722EC9B7B57BD9819BAAC01556002D88"
                + "3B8BD842C3EB3BCC4A54B4D0B1DB32ECEBA8DD668D67C859A6EB0BAE293082054106092A864886F70D010701A0820532"
                + "0482052E3082052A30820526060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D010C010330"
                + "0E0408482E476C7712FD7202020800048204C8AF5357D64578320D963917F44B2B7714334AAE6571554F03A599913275"
                + "4BA03892316159385225C4EEA7C958091BC8A32A9433AECA3A07F791ACE431217F0DFBD53DC15D257069B99DA04EF719"
                + "892004FD307E151EBB359C1D69AE8FF78A1CC1654470B0862CDAC1AED521608C29AA8D30E81A4015D54F75421B9BDB29"
                + "5036D79E4535F28D4A2ABF4F203BC67065A709AEF6EAF4D6A3DC7387CB3459D82399F4797CE53FD2BD0A19B1A9566F74"
                + "246D6B6C50BD2777B6F6DE1A8C469F69D7EBF230018D56DF4C1764862CD982E81F56A543DA4ADB63EF8612A1BB561471"
                + "56035541B0B41F06BBE2CD47DC402A75023558205870060438CF99D8BFC7CAADDE0583311FE4B854051C83638420BC5E"
                + "93F999E67EDBBC266F519609E2BE9FC1BC3C7FEE54DBAB04DAC8A94BB347F3DC28DDAB7D28DD3BBFFB44C84E6F23A8E0"
                + "1CAB36382723DB94CD8973A303D8D4C7A22B9F811C07ED9A78E06135E0426FC93BB408F1DC915DF4ADBF048D22C201D8"
                + "0FDC0EF942D1E2AC0F39F8A95BA849C07BB0DA165B3F0317478870F704B8A34B7D5816BC4F8CA0C6BDB5346F2211416C"
                + "79D7117AD1B86E44E0BC0C3793F9895638E5B7A2A5B957E0E691819AC7FA1F05E8D05ED99E4E286C96E3E31DF99633E8"
                + "CB73CA135109AE727CB047641726A1599492B6F3E8E62195A79239279B2A9FBF47B31FEFF3C20DEC2DFBDB0CE98B183D"
                + "BA773561DEE404BA1A5BEF5AB9729DBE22FB1C17EFD4D3AC81F04F49F9855CEACECB202090A1290C10E9D676F0658F3D"
                + "E4C43DCD5A17B88881893DA87060C5F56D5CC9A92E6B1A47A6D16FB32C09925606F6D5C7CAFBC7A82D8E441A05DFBEE0"
                + "BEC92D89264B62D5DECC342D29D9A7727BBDE4E63EEB7CAED7C76953F6AC8CB570619C7607B753FD46889C76D29C9AC6"
                + "6F56CB3848323FA9CD16578EA5C6D876AE63D95F65E2CDEF68A1CF3D2FC3DF91D0055B0CDBD1510E291C0E7AC6EAA0D2"
                + "AB5E8FAD44108C94A592251447926DB7139BC2A433D61711C6DA5EF82A8E18CEBF99AF753E33FFF65126B7D3D3D09FF0"
                + "C50EFF7822FA8797BAC52383B94E8FE602E62577994ACA6A2150F60A31CA0E79FE6DF3405D6918EADC2231558FB29045"
                + "034EB9DA9FB87BD71996C6AB6EA71A70EBFBC99BC292338A363176516C14EC92E174C59C6BE82F5BC0296525109C9A7F"
                + "C9D9E654955992A5C9EDFD39ED9889BEAF105B2EF62B041789F20A6AB26563FCFA1A1482EE2A20E8C1A2F0931ACBA7F8"
                + "756EE4C9119D29817ACA7D2B81FE736FD7A33D20EC333AC5123D29345647B734DB24B5C56B4576ABBF9B02F782DDE0B4"
                + "BA277080F28F3D86DEC35F0F19B2B5DB0BD7A59B7C4B2BAE08E8584449BD3685F371F6A24F5F91EA6843DC6ABA89976E"
                + "589511EB23A733D23F6CE076C952E9E7190ED78D5A34387F93418A87CB02270941F19DD35E1DB5836E67296A7F28A5EB"
                + "8F32DA73EA7A47D5CEB292E767468CDF938A44B3CEEE6276A34705606A8F7496D7310DA1A0468A604B8A9E7AB50450A6"
                + "DFE0C4540CEA058CD7919E823D8A32FB811C6BF8754C65D7A9723018ADE95AED5C30978A8DBA185CF0BA36346456CD3E"
                + "15C511FAD71B445DDFA7C5455A3597FE536E3BB87518C0725D6BE673D05DC5E74B4FF442711D242A37D0CCB88E6D19BD"
                + "6B7299207D7036EB87D5E86189768CB14AE4F8886BB5AB7864BDA9757D0C26BFFF3FAA4001258557D394313125302306"
                + "092A864886F70D01091531160414080FB9AAB81BD67FD85C2186B359054CEB13D2D730313021300906052B0E03021A05"
                + "0004142C205F0B1E9B99B0ED14E83F13D84BC683F66D3B04080D22E45D6A657CC602020800").HexToByteArray();

            public static byte[] s_Rsa2048SignatureOnlyCer = (
                "3082032C30820214A003020102020900E0D8AB6819D7306E300D06092A864886" +
                "F70D01010B05003038313630340603550403132D54776F2074686F7573616E64" +
                "20666F7274792065696768742062697473206F662052534120676F6F646E6573" +
                "73301E170D3137313130333233353131355A170D313831313033323335313135" +
                "5A3038313630340603550403132D54776F2074686F7573616E6420666F727479" +
                "2065696768742062697473206F662052534120676F6F646E6573733082012230" +
                "0D06092A864886F70D01010105000382010F003082010A028201010096C114A5" +
                "898D09133EF859F89C1D848BA8CB5258793E05B92D499C55EEFACE274BBBC268" +
                "03FB813B9C11C6898153CC1745DED2C4D2672F807F0B2D957BC4B65EBC9DDE26" +
                "E2EA7B2A6FE9A7C4D8BD1EF6032B8F0BB6AA33C8B57248B3D5E3901D8A38A283" +
                "D7E25FF8E6F522381EE5484234CFF7B30C174635418FA89E14C468AD89DCFCBB" +
                "B535E5AF53510F9EA7F9DA8C1B53375B6DAB95A291439A5648726EE1012E4138" +
                "8E100691642CF6917F5569D8351F2782F435A579014E8448EEA0C4AECAFF2F47" +
                "6799D88457E2C8BCB56E5E128782B4FE26AFF0720D91D52CCAFE344255808F52" +
                "71D09F784F787E8323182080915BE0AE15A71D66476D0F264DD084F302030100" +
                "01A3393037301D0603551D0E04160414745B5F12EF962E84B897E246D399A2BA" +
                "DEA9C5AC30090603551D1304023000300B0603551D0F040403020780300D0609" +
                "2A864886F70D01010B0500038201010087A15DF37FBD6E9DED7A8FFF25E60B73" +
                "1F635469BA01DD14BC03B2A24D99EFD8B894E9493D63EC88C496CB04B33DF252" +
                "22544F23D43F4023612C4D97B719C1F9431E4DB7A580CDF66A3E5F0DAF89A267" +
                "DD187ABFFB08361B1F79232376AA5FC5AD384CC2F98FE36C1CEA0B943E1E3961" +
                "190648889C8ABE8397A5A338843CBFB1D8B212BE46685ACE7B80475CC7C97FC0" +
                "377936ABD5F664E9C09C463897726650711A1110FA9866BC1C278D95E5636AB9" +
                "6FAE95CCD67FD572A8C727E2C03E7B242457318BEC1BE52CA5BD9454A0A41140" +
                "AE96ED1C56D220D1FD5DD3B1B4FB2AA0E04FC94F7E3C7D476F29896224556395" +
                "3AD7225EDCEAC8B8509E49292E62D8BF").HexToByteArray();

            public static byte[] s_Rsa2048SignatureOnlyPfx = (
                "308209E3020103308209A306092A864886F70D010701A0820994048209903082" +
                "098C308205BD06092A864886F70D010701A08205AE048205AA308205A6308205" +
                "A2060B2A864886F70D010C0A0102A08204FE308204FA301C060A2A864886F70D" +
                "010C0103300E04083EF905F0EA26FBF7020207D0048204D82297B5546DA6CC49" +
                "BD8C1444E3FE1845A2C9E9BDB8B83E78235DF4ADF7A97496A62D31D4EEEB76B0" +
                "71C0B183ACC3663272F88CF4F31E2E00D76357C0A051B8D6E0BB0BCF4CCDD064" +
                "CCBAF546EABA80DA56CD11FE952C61154792559D65F26B0476CF7A5FDB8CC794" +
                "B89F6ACD50003459054FE82C48D8791B226A0EEEC01F048AC3CE716C9F3BB313" +
                "D64BEBBF0037D83133DD9C15D04F15BB11652D793B613A68AFE580245724E5D1" +
                "110040B332B5C39BE04086BA4DFC58E905BC2FE8B3C696181E2879AF197EE24D" +
                "91D8AD67013F14C4864C8D0FB19C134B766CF3E48B8C9E363A11EB19F1E82E74" +
                "D25EDD96517D64A94314B40C11651030D561E742E63856E8D1A3EE9FDFD6CF64" +
                "7140CFC354AE7EA1D14C157C2985F82D54296F7D3DE456AF7513F5F30A0421E4" +
                "3A9DAD6DF8A2BF69005B35CA8066F80755D848DA73EF03BC0CC129C5799911D9" +
                "3A1ED43F8E76732AF56FD62DC6D0B0DBA6AAC6DCDE77D0E8AC9F6A5EB5A02B61" +
                "BC477706D4F1873240AB45E0291EF21E6034D48F1AE8EB139DE7ACD8B8A821E6" +
                "70B395C3EC4B0E75C34BF0067F052FCED835CAC1F17C3FBEA2FC9FD281FCDE21" +
                "D5B27CF31A07E90164A979ACEF0E1C67DBA6C33082E9B189D4BDA2D2D504776F" +
                "455843437BDF10D4AF48639EC03344BCC36EFEA7CDE08D7F94DDEAF98BCB5D65" +
                "207AE0C349EECE3032DE19F3B337F29A3457AA0AAF4306FA8301619AB01B7235" +
                "BE16CB93728E142DAA6C1CBBCC5BD82D913596994DA40FB916CF2DB5FBCC20CF" +
                "E893DC62BBC5FC59E00BC0A704A9DB25BBF9291971F2377FC1A20F2C954869DB" +
                "6FFCC90C625AABE97ED4CF7C0209D39AD780003C437152D636ACB3B484C46885" +
                "DC1584733D2153A3F9B968F12CDD5937CDF9DD2581D72EE67C83AE2530197AA7" +
                "C6110613BEFF0B75E586C28394EA8EBCF7F9DB133295B33DC86C8DBA92BF8BD1" +
                "ADCAF8D2CD2E018B08D59FF1C30A13484AB11468F7DCEB1FE53A6DAF309B0510" +
                "7772CB735314A5B2F053E60A653F0496BCB9CADF5E50339A4D2EF2382056B768" +
                "558EB9230D996C636E6D29664F92F70A088DE3EE4EC4BBD8A9C4C98C7892D122" +
                "28806622B87E581A321AD835B8F4B964A17B5BE6D9DA50133D494732A41884E2" +
                "9E891FE2D40ACCFD585C8BF626C1E8A412D2EE7CDE060E2CCDA826BF79D80F1B" +
                "F6B8400473BCE0C19D03ACF55D1FAA994C04A8CD11D49743B1F45F48DFFDD701" +
                "18B5FA82ECDF67714F5DE5D3D3DDDCB76ED0EA6A6E151665A4AA351DB1A99F8C" +
                "7502D3795C2C358CCA589C390C1F4810615130B91BA42A85E77FA37197E1B083" +
                "FE1246B067C6444D49B369D45B130A6D7B463C3F0459EB41D68009CABD2F5C60" +
                "49DB706FA742C9773FB5791AF123FBE485E05F87759ADD25281BE337B6095EC3" +
                "4EFF9FC692798FB4217EF4B2B59902D930F28181933FAA278C041123CAE3CA63" +
                "6DFD3AD4E04EB751A30D50C26288EA4D01C7B323E4FD6387F88E020BC433BF60" +
                "C4406398C44EA5C7A6EB24134B0811E4F94DFAF5553172306FA5543C254E7E04" +
                "DEEC84DBF9FAF7BFEA8D61E094CBB18DD45C5BAB9199DD719F9A305E205605CC" +
                "671DCD566FEBA2C8F4C1A445625C4F42D1CFE32087F095591798D1D48DA46DE9" +
                "230F5102B56A1EF879D48936D5331D6B3D9F1B564CF08FD3C641CFF3B02CB4FC" +
                "8995E5EC5DD1D183704940C02DEA7430FD594E54800DCC74B7732731C63FBBA2" +
                "A2F6DC031174390A74781D352B09FB4F318190301306092A864886F70D010915" +
                "3106040401000000307906092B0601040182371101316C1E6A004D0069006300" +
                "72006F0073006F0066007400200045006E00680061006E006300650064002000" +
                "520053004100200061006E006400200041004500530020004300720079007000" +
                "74006F0067007200610070006800690063002000500072006F00760069006400" +
                "650072308203C706092A864886F70D010706A08203B8308203B4020100308203" +
                "AD06092A864886F70D010701301C060A2A864886F70D010C0106300E04087CB7" +
                "E0256AD1AD80020207D0808203800C21CEBEF9F3765F11A188B56702050E3DCA" +
                "78AA27123654D066399DD56E62187C89A30940B5B63493950EEFA06C04B5CAF0" +
                "329143AF30EE0B47406E49D4E6241817986F864780B743B58F03DF13523F5C01" +
                "C889046356623AFA816B163E57A36672FAC9CA72294B2A17F75F5ADB1A4CBDB7" +
                "B3F5C33C643DA0CC00CB79E54FAB25D1881B81C03BA5762BAA551A7E8BA38144" +
                "353B07285B288BC2747F75B7AF249040C338CFC585D0B1CECFED46BCAE7FAF09" +
                "60BB3EE996E30E626CB544A38393BC7DFDB7A27A21A6CF09332B544F448DF5B3" +
                "31E000F7CCD5CE5C8E8765A2339919C713352FCD30FA52B994C25EA95E548C4B" +
                "5EC23B3BDEC7342D0676B9227D3405758DBA5BD09F9253791FAA03F158F04848" +
                "D5073DD240F466F57770353528B3AE83A626F33D05BD1BBB4E28CB067FFAA97D" +
                "B4C79EEAAFB4B30BE738C1AA5DB1830B3968CDF6BAF778494AE40EF003DCDA54" +
                "486E9952EB44628385E149C348E0E431928B85608622B994CF43433DA9C19482" +
                "360121560E53E85FE7CBB7C31E27AD335BC247F284EAC3CA94C30DBB4DF2AB02" +
                "DF1154626838240213D910D5B7476A025CA7565CECBA0051320FC7EECD6C74FF" +
                "505566F75804D1E2BD2B0181B235CE911EAD9260C0799C817F956AE290E00EF0" +
                "997F7B6BD059B315915D580CF0C019A23A6D4993F6E8B8106A1AB6CE1991B609" +
                "1B42B6D33EE01EC96CB475430365E9C710C5EB4C6010260D12108022449F7E6D" +
                "1A2F28838304DB2A60B9FF714FC887579A4CDC139DAF30A18D3910D82313CCB1" +
                "FA43A8930E0F10DE24652AC1E5B797084BEBDB8AB5FA6DCE03E44ABF35EDEB1A" +
                "FFEAD3F7C9CB342CCA2882D945EB52C20DC595FA10161866EB9426281CF13341" +
                "311B59FDE8E69F9B853117740D92F4AC1B2E4597D41B8A097E1DAA688FFF1C5C" +
                "846DF96CA75224EC26F4FF328164F5D1EC06B697211BDB42E6C97EB294A5798C" +
                "0FCE6104C950A5207F74EC0DED8AEE10463EF2D9ACD7473D2BE48EBBF0A550B9" +
                "AA19A465147B378B078229E8804918136633D7FCE5340AC61A1418D7D9BB18D1" +
                "98B7B7866C4D7DC562B1F93F3F322484BDDCEB23680B8EB9904EC783D5CD7177" +
                "CFE9CA9D1893104E97760E871DE0907D4BDF6263E7BB0F47414AF31E377C7447" +
                "B881E68AE3E80D06597F12D5EF5ED861D055D494D89F04A70800DA3FD4E53877" +
                "87FBEED7B772E3A24E7F4832A956FEC0C81847C68373ED4760ABF542F77DC794" +
                "249519BDDF5F846EB8C5078BCC053037301F300706052B0E03021A0414461F5B" +
                "19C6933240012EFEB95F734C648CCD13460414FA1743400686D25BA1CB28D736" +
                "F2B1ED97699EA4").HexToByteArray();

            public static byte[] s_dsa1024Pfx = (
                "308206EE020103308206B406092A864886F70D010701A08206A5048206A13082" +
                "069D3082043706092A864886F70D010706A0820428308204240201003082041D" +
                "06092A864886F70D010701301C060A2A864886F70D010C0106300E04084AF212" +
                "89D5D7E2E702020800808203F0DECCF218AC91F26BAB026998AB77C7629D20DB" +
                "E2FB7022A3C4A1CECD743C0F932E944AE229DAFB61AD76C4DEB6995DF4F4BA01" +
                "2DBAD5C63A4C846E0807FCA0BC4A162CDFBAB4B3C4D304F473B3ACC1D268436E" +
                "F537DAE97ECC3C634C8DF2A294CC23E904A169F369021A0C024A03DE98A65B0F" +
                "3F14D6910525D76AD98B91E67BB7398E245CF48A4D2A5603CFCCF4E547D7EDAB" +
                "669D9A8597C6839119EB9FD932D1E4BA8B45D3317186CDA2EFF247BCFD64A5CA" +
                "ED604BF7033E423CC21CEC6454FE3B74E03A26C51A1C3519CE339FBE9F10B81D" +
                "DF6A0AAB4F8166D90B6F52B3439AB4B5273D0A506E3E01869F8FEBD1521EF8E5" +
                "BFB357FA630E3C988926EF3ACC0A0F4176FE8A93337C1A5C6DEAB5758EC2F07C" +
                "11E8B2495ECDE58D12312CCCA2E8B2EE8564B533D18C7A26A9290394C2A9942C" +
                "295EBB0317F5695103627519567960908323FFE6560AD054C97800218A52F37A" +
                "DDE4E7F18EF3BF3718A9D7BF57B700DBEB5AB86598C9604A4546995E34DBABBB" +
                "6A9FB483A3C2DFE6046DFD54F2D7AC61C062AF04B7FBAC395C5DD19408D6926A" +
                "93B896BFB92DA6F7F5A4E54EDBE2CFBB56576878150676ADB0D37E0177B91E0D" +
                "F09D7B37769E66842DD40C7B1422127F152A165BC9669168885BA0243C9641B4" +
                "48F68575AA6AB9247A49A61AC3C683EE057B7676B9610CF9100096FC46BDC8B9" +
                "BAA03535815D5E98BA3ABC1E18E39B50A8AF8D81E30F2DFD6AF5D0F9FC3636AB" +
                "69E128C793571723A79E42FC7C1BD7F39BD45FBE9C39EEB010005435BEC19844" +
                "2058033D2601B83124BD369DADB831317E0B2C28CE7535A2E89D8A0E5E34E252" +
                "3B0FCEC34FF26A2B80566F4D86F958F70106BF3322FA70A3312E48EAA130246A" +
                "07412E93FDE91F633F758BC49311F6CBBAEC5D2F22AFCD696F72BC22E7DE6C00" +
                "3275DFEC47E3848226FE9DBA184EA711E051B267C584749F897EFE7EAFD02F1D" +
                "BF3FD8E882474CA1F45509EF2E7B82F35B677CB88ED42AF729848EE2B424B0CE" +
                "2E9AAC945BABA550C20D5B25075A30FE70D8CAA5A527A35F1DF17BCCB91930C1" +
                "7120C625667120E0806C2B51EDFF540F928BD555FB48DBCB83CCCE0C385E78C8" +
                "65BE715AE6F8BE472E5FC187EBE3FEFD8D7FE62D4DB2EE61F42D24D81FAA9179" +
                "0FB17E8EBC8E219B6F9E039F5AB3BC4870821D474B36C8F8D0583D9DC06E4383" +
                "D03424420B8C8B26276877166A0F51E22F0D8FA60A070CFBD47EAFBC717C879C" +
                "B5A1EA69C4C2A38F26A1EEF96A0C32BFCECCE4EA97E90A425066B1DD0891353F" +
                "766EB9F2BFA2563A815DAF3639EBB147E1E8757A6BFAB902C4A8F037AD47E03F" +
                "AF2E019FCF6CA7430BDFEA4B45B28ED746BB90E09BEF7B370A75E7924BBA0920" +
                "25FE654A9A197A5B8BBBE43DC7C892FF14E75A37EB97FC489AB121A43E308202" +
                "5E06092A864886F70D010701A082024F0482024B3082024730820243060B2A86" +
                "4886F70D010C0A0102A082017630820172301C060A2A864886F70D010C010330" +
                "0E0408ECB4D1550DA52C6302020800048201509322DC0193DD9E79ADAFD38827" +
                "AD6DE9299327DDDF6E9DF4FB70D53A64951E4B814E90D2A19B3F4B8E39A2F851" +
                "A3E5E9B9EB947DD248A3E5F5EB458F3323D4656709E97C6BD59238C4D1F26AB6" +
                "7D73235FAE7780D98705957B6650AC0DE3E2D46E22455D0A105D138F16A84839" +
                "14EDDF5C518B748558704ED3AE4A8C4914F667BBDE07978E4A4FC66194F6B86B" +
                "AB9F558EDE890C25DFB97C59653906CC573B5DEB62165CFF8A5F4F8059A478EB" +
                "F6FED75F1DACDC612C2E271E25A7083E15D33697270FD442D79FFCB25DB135F9" +
                "8E580DC9CE14F73C3B847931AF821C77718455F595CA15B86386F3FCC5962262" +
                "5FC916DDB4A08479DCB49FF7444333FA99FBB22F1AEC1876CF1E099F7A4ECA85" +
                "A325A8623E071EEA9359194EEE712F73076C5EB72AA243D0C0978B934BC8596F" +
                "8353FD3CA859EEA457C6175E82AE5854CC7B6598A1E980332F56AB1EE1208277" +
                "4A91A63181B9302306092A864886F70D01091531160414E6335FA7097AB6DE4A" +
                "1CDB0C678D7A929883FB6430819106092B06010401823711013181831E818000" +
                "4D006900630072006F0073006F0066007400200045006E00680061006E006300" +
                "650064002000440053005300200061006E006400200044006900660066006900" +
                "65002D00480065006C006C006D0061006E002000430072007900700074006F00" +
                "67007200610070006800690063002000500072006F0076006900640065007230" +
                "313021300906052B0E03021A0500041466FD3518CEBBD69877BA663C9E8D7092" +
                "8E8A98F30408DFB5AE610308BCF802020800").HexToByteArray();

            public static byte[] s_dsa1024Cert = (
                "3082038D3082034AA003020102020900AB740A714AA83C92300B060960864801" +
                "650304030230818D310B3009060355040613025553311330110603550408130A" +
                "57617368696E67746F6E3110300E060355040713075265646D6F6E64311E301C" +
                "060355040A13154D6963726F736F667420436F72706F726174696F6E3120301E" +
                "060355040B13172E4E4554204672616D65776F726B2028436F72654658293115" +
                "30130603550403130C313032342D62697420445341301E170D31353131323531" +
                "34343030335A170D3135313232353134343030335A30818D310B300906035504" +
                "0613025553311330110603550408130A57617368696E67746F6E3110300E0603" +
                "55040713075265646D6F6E64311E301C060355040A13154D6963726F736F6674" +
                "20436F72706F726174696F6E3120301E060355040B13172E4E4554204672616D" +
                "65776F726B2028436F7265465829311530130603550403130C313032342D6269" +
                "7420445341308201B73082012C06072A8648CE3804013082011F02818100AEE3" +
                "309FC7C9DB750D4C3797D333B3B9B234B462868DB6FFBDED790B7FC8DDD574C2" +
                "BD6F5E749622507AB2C09DF5EAAD84859FC0706A70BB8C9C8BE22B4890EF2325" +
                "280E3A7F9A3CE341DBABEF6058D063EA6783478FF8B3B7A45E0CA3F7BAC9995D" +
                "CFDDD56DF168E91349130F719A4E717351FAAD1A77EAC043611DC5CC5A7F0215" +
                "00D23428A76743EA3B49C62EF0AA17314A85415F0902818100853F830BDAA738" +
                "465300CFEE02418E6B07965658EAFDA7E338A2EB1531C0E0CA5EF1A12D9DDC7B" +
                "550A5A205D1FF87F69500A4E4AF5759F3F6E7F0C48C55396B738164D9E35FB50" +
                "6BD50E090F6A497C70E7E868C61BD4477C1D62922B3DBB40B688DE7C175447E2" +
                "E826901A109FAD624F1481B276BF63A665D99C87CEE9FD063303818400028180" +
                "25B8E7078E149BAC352667623620029F5E4A5D4126E336D56F1189F9FF71EA67" +
                "1B844EBD351514F27B69685DDF716B32F102D60EA520D56F544D19B2F08F5D9B" +
                "DDA3CBA3A73287E21E559E6A07586194AFAC4F6E721EDCE49DE0029627626D7B" +
                "D30EEB337311DB4FF62D7608997B6CC32E9C42859820CA7EF399590D5A388C48" +
                "A330302E302C0603551D110425302387047F0000018710000000000000000000" +
                "0000000000000182096C6F63616C686F7374300B060960864801650304030203" +
                "3000302D021500B9316CC7E05C9F79197E0B41F6FD4E3FCEB72A8A0214075505" +
                "CCAECB18B7EF4C00F9C069FA3BC78014DE").HexToByteArray();

            // Password: "Test"
            internal static readonly byte[] ECDsaP256_DigitalSignature_Pfx_Windows = (
                "308204470201033082040306092A864886F70D010701A08203F4048203F03082" +
                "03EC3082016D06092A864886F70D010701A082015E0482015A30820156308201" +
                "52060B2A864886F70D010C0A0102A081CC3081C9301C060A2A864886F70D010C" +
                "0103300E0408EC154269C5878209020207D00481A80BAA4AF8660E6FAB7B050B" +
                "8EF604CFC378652B54FE005DC3C7E2F12E5EFC7FE2BB0E1B3828CAFE752FD64C" +
                "7CA04AF9FBC5A1F36E30D7D299C52BF6AE65B54B9240CC37C04E7E06330C24E9" +
                "6D19A67B7015A6BF52C172FFEA719B930DBE310EEBC756BDFF2DF2846EE973A6" +
                "6C63F4E9130083D64487B35C1941E98B02B6D5A92972293742383C62CCAFB996" +
                "EAD71A1DF5D0380EFFF25BA60B233A39210FD7D55A9B95CD8A440DF666317430" +
                "1306092A864886F70D0109153106040401000000305D06092B06010401823711" +
                "0131501E4E004D006900630072006F0073006F0066007400200053006F006600" +
                "7400770061007200650020004B00650079002000530074006F00720061006700" +
                "65002000500072006F007600690064006500723082027706092A864886F70D01" +
                "0706A0820268308202640201003082025D06092A864886F70D010701301C060A" +
                "2A864886F70D010C0106300E0408175CCB1790C48584020207D080820230E956" +
                "E38768A035D8EA911283A63F2E5B6E5B73231CFC4FFD386481DE24B7BB1B0995" +
                "D614A0D1BD086215CE0054E01EF9CF91B7D80A4ACB6B596F1DFD6CBCA71476F6" +
                "10C0D6DD24A301E4B79BA6993F15D34A8ADB7115A8605E797A2C6826A4379B65" +
                "90B56CA29F7C36997119257A827C3CA0EC7F8F819536208C650E324C8F884794" +
                "78705F833155463A4EFC02B5D5E2608B83F3CAF6C9BB97C1BBBFC6C5584BDCD3" +
                "9C46A3944915B3845C41429C7792EB4FA3A7EDECCD801F31A4B6EF57D808AEEA" +
                "AF3D1F55F378EF8EF9632CED16EDA3EFBE4A9D5C5F608CA90A9AC8D3F86462AC" +
                "219BFFD0B8A87DDD22CF029230369B33FC2B488B5F82702EFC3F270F912EAD2E" +
                "2402D99F8324164C5CD5959F22DEC0D1D212345B4B3F62848E7D9CFCE2224B61" +
                "976C107E1B218B4B7614FF65BCCA388F85D6920270D4C588DEED323C416D014F" +
                "5F648CC2EE941855EB3C889DCB9A345ED11CAE94041A86ED23E5789137A3DE22" +
                "5F4023D260BB686901F2149B5D7E37102FFF5282995892BDC2EAB48BD5DA155F" +
                "72B1BD05EE3EDD32160AC852E5B47CA9AEACE24946062E9D7DCDA642F945C9E7" +
                "C98640DFAC7A2B88E76A560A0B4156611F9BE8B3613C71870F035062BD4E3D9F" +
                "D896CF373CBFBFD31410972CDE50739FFB8EC9180A52D7F5415EBC997E5A4221" +
                "349B4BB7D53614630EEEA729A74E0C0D20726FDE5814321D6C265A7DC6BA24CA" +
                "F2FCE8C8C162733D58E02E08921E70EF838B95C96A5818489782563AE8A2A85F" +
                "64A95EB350FF8EF6D625AD031BCD303B301F300706052B0E03021A0414C8D96C" +
                "ED140F5CA3CB92BEFCA32C690804576ABF0414B59D4FECA9944D40EEFDE7FB96" +
                "196D167B0FA511020207D0").HexToByteArray();

            internal static readonly byte[] ECDsaP256_DigitalSignature_Cert = (
                "308201583081FFA003020102021035428F3B3C5107AD49E776D6E74C4DC8300A" +
                "06082A8648CE3D04030230153113301106035504030C0A454344534120546573" +
                "74301E170D3135303530313030333730335A170D313630353031303035373033" +
                "5A30153113301106035504030C0A454344534120546573743059301306072A86" +
                "48CE3D020106082A8648CE3D030107034200047590F69CA114E92927E034C997" +
                "B7C882A8C992AC00CEFB4EB831901536F291E1B515263BCD20E1EA32496FDAC8" +
                "4E2D8D1B703266A9088F6EAF652549D9BB63D5A331302F300E0603551D0F0101" +
                "FF040403020388301D0603551D0E0416041411218A92C5EB12273B3C5CCFB822" +
                "0CCCFDF387DB300A06082A8648CE3D040302034800304502201AFE595E19F1AE" +
                "4B6A4B231E8851926438C55B5DDE632E6ADF13C1023A65898E022100CBDF434F" +
                "DD197D8B594E8026E44263BADE773C2BEBD060CC4109484A498E7C7E").HexToByteArray();

            internal static readonly byte[] ValidLookingTsaCert_Cer = (
                "308204243082020CA003020102020401020304300D06092A864886F70D01010B" +
                "05003029312730250603550403131E4578706572696D656E74616C2049737375" +
                "696E6720417574686F72697479301E170D3138303130393137313532355A170D" +
                "3138303431303137313532355A302C312A30280603550403132156616C69642D" +
                "4C6F6F6B696E672054696D657374616D7020417574686F726974793082012230" +
                "0D06092A864886F70D01010105000382010F003082010A0282010100D32985C3" +
                "0E46ADE50E0D7D984CC072291E723DC4AA12DF9F0212414C5A6E56CBB9F8F977" +
                "73E50C2F578F432EDADFCA3A6E5D6A0A3ECFE129F98EAA9A25D1AB6895B90475" +
                "AD569BF8355A1C4E668A48D7EAD73206CCA97D762EB46DA75CF189E2EC97A8DE" +
                "BA8A4AF9CFAB6D3FD37B6EB8BBED5ADA192655831CFDAA8C72778A314992AB29" +
                "65F3860B74D96DEB2E425216D927FCF556B241D43AAF5FA47E2BE68592D2F964" +
                "F5E0DE784D0FAD663C3E61BD4A4CF55B690291B052EC79CEA9B7F128E6B8B40C" +
                "5BADCDB8E8A2B3A15C7F0BD982A1F0C1A59C8E1A9C6FC91EE9B82794BA9E79A8" +
                "C89C88BF8261822813E7465B68FFE3008524707FEA6760AD52339FFF02030100" +
                "01A351304F30090603551D1304023000300B0603551D0F0404030206C0301606" +
                "03551D250101FF040C300A06082B06010505070308301D0603551D0E04160414" +
                "EEA15BD77DF9B404445716D15929ACA4C796D104300D06092A864886F70D0101" +
                "0B0500038202010046AC07652C232ED9DB6704C0DD318D873043771CD34CA966" +
                "91C04E5B7A311805E965957E721358B2ABA55A0445864251263E09BFE2EAEB64" +
                "43EA9B20F05502873B50528E48FF5FD26731BEA1B13CA3E3A3E3654E6A829675" +
                "B9D7944D734C48D36E19B88B77320A5096BF7462E1B4D69F1DC05F76A9417376" +
                "D5127027B1BA02380684DCEB75A4B14FA36B2606DDBA53697CE7C997C2FF13E4" +
                "66234CBE07901DF33A037F323FEF8C4229F2B7D030BAC88B4C3D84601576B0DE" +
                "32F3116C6DF7E9AA29028289E0CCA89F8B809310C37E9BD514536131D9E66AF0" +
                "6B1F36BAD55C9F9D6D1393CF724D75D3441AD67491AA76C8C31AADE22209831F" +
                "C079B49408ACC2C0D975EF8BEE3E0F6A01DA4DFC6045433BA6B1C17BB0E3E181" +
                "22770667CBD6197413569470BF0983BF358C6E09EC926631B0A2385D3BF9D9F3" +
                "0B5314170865D705CA773652BD66E1B3C112D7DA97CDFB9447FBFCD4DF674AB8" +
                "A6F430760276B7D8439BE40961D0B7F9F2B7AC1D291C0E67C1789FE038B6652D" +
                "24FCAF0A49CDB1E61FBA146AEFA3D934BF3B6AE8A5703CCA80AA500B56DF93FA" +
                "931E2D71E292042342CFB073431AF0AA1ACC0B5F53EBF66CFECD52EB08B11628" +
                "000F5EA01AB1D8E89F219178DB67B2CD68AFC2C0C248D8A65FD9AE1A0DBFF84F" +
                "3BBF2077EBFB373F6ED8D6C7CEA7BFDF0425494078F293949496B0BEAF63CAB5" +
                "62C297590A737174").HexToByteArray();

            internal static readonly byte[] ValidLookingTsaCert_Pfx = (
                "30820AAE02010330820A6E06092A864886F70D010701A0820A5F04820A5B3082" +
                "0A573082059006092A864886F70D010701A08205810482057D30820579308205" +
                "75060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E040809F1CB8B4255EADD020207D0048204C83EF1251C78FF3815" +
                "CDB0648F20FF20FA0437E5AAC5CB91C0D547755113724A913C02427A12A878E3" +
                "B161F388F4DA9AEFBBA5FEB274CEF3A35CC2EC4BFE32442E5B6624888D41FC3B" +
                "EA02BBEDA034BB196A3FA41E23DCEB6F6E906FD86FED066D547518FD628C66F0" +
                "1AA4B38F3DDD94D63A5B336B587BAC4D2B739EF096A902ECC4216EC284099F10" +
                "C93785AFC3939A44C22FD027E4E643B03641FB3B76B21DB632D8522A365A495D" +
                "5AC74CF7765E294CEC55C73F6A4BB45ABD214D7AECBC3735DA41D8FC66CD5C34" +
                "54F354E16084D0E1984B20423219C29CAE0FDCD16A16C5BF17DB23DD8F2B1C1B" +
                "DFC9008B647D2FD84E4EC7952BFDF4EA0F0A13D57CD530109BFBA96DD3F37F01" +
                "7F7BA246C35A9D5C0A2294A2EEFE35B29542A094F373B6FFECE108D70CEDB99C" +
                "A7172B17C6C647CD6614D3FAE0C02B3D54062FD8392F152AB1B46D1C270A9F19" +
                "A48A393CCF22EC3DA958C35A8A6A3E7CFFDC2C54090F844B3B73C3CE7F7EF26C" +
                "982567ED621FDB93E402FC145E6D7E8D7F2F9C56F870823C98B81586F34C7C96" +
                "CBAA5A67544964FA1BD70B9C5E8284ACF22FFC69BF62E389C5678E44CB10D6C3" +
                "D3F73DA7BF12B44D74C0233C140ECC93C5F0118C8F0D9B6FFDFB23966ADC070C" +
                "8DBFAFE78AE1E05F8FA7C75A22FBF7A8458B9C61B1F2BF2AD2F0A6C6D15AAD55" +
                "D960F173AC3D9890CAF50228796FAD6C7EAB1C8A3C8E9E42E4A4DA99C4A67FB3" +
                "E2AC958AD989508538C96697A0CFBEEB38E9CF27DAE3AB9492729A205CB25340" +
                "85CA0D580DCD49E88DA7950F99CD05523885B5257BD302E1B449F03941BD0DA1" +
                "ECCAE6A22BC9F7E446F7F6FD4EE850CA3BDD7338F705D93C2F196712250BCB1D" +
                "A18C9661E195486768515BC2761A66F61770589A62D172DF8EC288549E86E518" +
                "7B04E1550154FF45060945BDA07014F14EB554A5A324F9B79DA192F79AB0745D" +
                "F30355DF580778773F2FFC76FB7B367EDBE173AC7F70698766DE6BB384A5C88B" +
                "66B163E8ABBF0AA44C4ED0D3D51D167E8BEFB2E71D36043ADB098BF2DADD409C" +
                "1F53F5546D1C8A1DC5E7BE566D14C494460161BFA7CB7E936790A81A45763004" +
                "689FA9BC33F31B4E86A152700D35B20380F87F4304D7850CA7BF583724328E0A" +
                "0D9B611B919975DF118B334D9DD96A46A21B00FC3B7FCCAFEA968FF030EA5D8F" +
                "9AD8624F668B2A7E433E54547EB89FB876A7E1AD2E9DAA38F30E938D8BCFB431" +
                "6E12FB8BEBF57FD0BF542E55386A6DEE074CFC6A33A55A57503CAB7F115DB875" +
                "C8EBF787670BE98DC03C65F2502D6F8D001ECC819BBB9C60BFC3A88DB8A117D9" +
                "9E09C13AC23324E15E5EE3C22B084D428FF2DFB37F917F7629F8A15093BB7777" +
                "B1AD8CACB4A5C6271E8B21A18DB95D6196E9EBD870521CA16930F2D1D43962AB" +
                "B8413016DA0117E10AB2622FC601DD08826429D8B8AE9BC6F15AE78392C36BC3" +
                "06FC19C90AD43BADD9AACDFA8CC16075529BFC8C34766C12783BF2F2E0B393CD" +
                "4F8F05D588B99218806D57CD5935E25DB2AE20DC4CDFD7F5111AF9A9EFE45349" +
                "42CAAA72F1F95636085FEC84BB98D209FD4222BC5F84DE5926A28FF7A5B7548A" +
                "B4FC3331431964454A0C532C205CF8D3872C561E83D805F7BD7DC811A0A90C9A" +
                "CB308E8F06AB739DCE97A840B4AFC0E884982CFC9B37897CF272ED1F46027101" +
                "BC97B11F04D64B07556DCFD5F609C5C9FB4B3F2AB345CAB46211EF0BE5ADD6BD" +
                "3174301306092A864886F70D0109153106040401000000305D06092B06010401" +
                "8237110131501E4E004D006900630072006F0073006F0066007400200053006F" +
                "0066007400770061007200650020004B00650079002000530074006F00720061" +
                "00670065002000500072006F00760069006400650072308204BF06092A864886" +
                "F70D010706A08204B0308204AC020100308204A506092A864886F70D01070130" +
                "1C060A2A864886F70D010C0106300E0408C10D9837801E56F8020207D0808204" +
                "78827979D15533A4A7210A435D0189C133B65B00407E85BC44368D066C65007C" +
                "256EE51A55E35BF8EE7C6FAC3D67FF7CF2031FECDCC356A51396B2977E044A79" +
                "E6C6CB8859E4AD49765468A4A467071292EFE47AEB39856FF8F00B5C6C6190EA" +
                "B20CC9A7C630C09E3F351ECB20CEC1BFE7BEB5A3FD534BAF8CDB658318A37279" +
                "269A11E8A87074FF0B111E2CFC493BD08D7887A7A222743B0C50E47F676F9B47" +
                "449F8FCBC6AE5F5A3456AE6BC3CB8A3CF28C59D0A16FE4336E0BFCA0AD74F95E" +
                "0F1010C3F698E16418E46B0059AB8F3DFD31FDB99132665CEC4CDAE8B6C1D0EA" +
                "9DADB8E591769261C27188CD5FF8D5C56E6866D85E1502254823940EC77096F0" +
                "6D3A261F49495AA60114BDDCA27C603F78314678CB08738FA2974DE03BE315F9" +
                "1FCA511446C68127211CF575948550DE4F7FDBF4AC31E395E12EBFD4BB99F470" +
                "498846940A92B6F85CF5D745B5230EAF994DD368FF55809F299213749CBABE54" +
                "C54D39B2165370DA43468409E73C55EF784D17496AAF8B6B5CC0F9FF234D7CF5" +
                "C84C248E91F8B08F74A8A953F60407940A2EB9576655CD9946D468DD6B3DB634" +
                "A30D4FE8E09C51AAB99C3DB5CC834A447EFBFDACEC9A49AFD675BCB247575B27" +
                "ADEEA9C5A6F2FCFF71A57EA99B467029C3E40D94D2849527FB90B9BF39294CC1" +
                "AD1F4CC18F75302FF241E05896A357BAAB88BDA8F7EB22B9AEFC5D7600154631" +
                "3C2E87452F99997CF4747CA936BD9199C61CEE5E61BBDB80C17978938F241D53" +
                "AAC97758FD1A77F3C7C28F88F1BDC7E74176CA41B73B7042A6C3C7AE918635E1" +
                "1EF4F2607588B0FF7C656541C32A1450AC71AEA42C14E94F0408A09902C30D10" +
                "ABE8396EDF5760991C2E02949EED06D091A2AB64827A53F29FA8B8B79036DA26" +
                "210A7C34EBE73590DAE990666E0F0F011EE8E41D08E16B28D49F30D8539EF18B" +
                "61860EE73D9AF98E4D3F17A50D06FC0592C218F38C33E0B526761D030E037B27" +
                "348CFDBB72C6B635BAEA497BA12618BFA47A9E358EE834FBA8A5FA9D30A66CFE" +
                "BFAF1E25FAA361F94433B40628E7423491BBA29116EA6ADDC6DFEFFA3AC2AD83" +
                "83C3EA05DE72B7CE357612A9C60CADCD42DECCD40438A9FC8E5D9413825178A1" +
                "9D67E7B13449F118A6B56A4F8A5DCC0AA05D0AB40B8FE0820A2399307331524D" +
                "B3389C97181EDC2ED2C5B72C1318B406CF99120CD784B85238488D1DA062478D" +
                "1EE6380F9954BE5442C1C00ABF8D284AF95937E2D515E2E8858F3C50516CA00F" +
                "E3F632BB05EE4AD63A6F2D72C7F2B06DC993866F6A740C1EF5159117437F646C" +
                "A47AA4BE5936C2F6BEF4C628A744FA2A2E2068752FF43BB9CB29C986D3A466FB" +
                "F3A107E8D610B8236E4E074CD6B32B50BF1C389CBDC2A92FDD19047450E04B4F" +
                "B963CA94860C1DF05D9FE1D202227E351BC2226572D6950D7E5242BE17DCC992" +
                "A989B7D13BAB9C8EC76FF5B8DED7696BBDAC058F6183F6F799365EA1DCC4F1B2" +
                "43A35027BA2F02E19DFCABEB564536EDA7B073519C60294C4AC25956D9F9DA60" +
                "3C5E4F335D7570299993460BDC5F20776F22D745A6B3F7E80EC69DA881AC72A4" +
                "D6B7AADF7EF19C77A2555C1CF8CE1CEB030EBF1C365D40C9C33037301F300706" +
                "052B0E03021A0414AACBB391FF9626295715807ED7DDEE57F716A5710414658C" +
                "344F4B20292DD9282953DAA4CB587AD48714").HexToByteArray();

            internal static readonly byte[] TwoEkuTsaCert = (
                "3082044030820228A003020102020401020304300D06092A864886F70D01010B" +
                "05003029312730250603550403131E4578706572696D656E74616C2049737375" +
                "696E6720417574686F72697479301E170D3138303131303137313931325A170D" +
                "3138303431313137313931325A3026312430220603550403131B54776F20454B" +
                "552054696D657374616D7020417574686F7269747930820122300D06092A8648" +
                "86F70D01010105000382010F003082010A0282010100C456AE596BB94EA55CE7" +
                "D51785F44223F940237C1F0A279533875427547BDC3944B73E8E6F4463800571" +
                "226147CEA3649972F96F128B673BCA6BBFD70B5178FE93D4DD7BE9E4D450AA0B" +
                "4D177F24DBCB2A7A13D7F10BABCE0E9AD3B853F01872196F6905F523E260555C" +
                "AFC5B158A82ED52D62BDA32142982EE8BB4E011E44622B59387B8A287F4DD7A1" +
                "5C783EAB5D4736CAB0E06A78EE50A7BA59DFE2C35DAEA0637FD581DB485ACA9A" +
                "57B94585A9E7D7BFAEE31F92F96DB5F95DDCE52B839BC06C30191014FE6804F0" +
                "CF4CA29412EB34D87303CE5FB49EC3E4B1A7CB001B6F117E1D5846A605ECF5D6" +
                "8FDC32EA972CC8521B823708518BBE59D15D10DB79990203010001A373307130" +
                "090603551D1304023000300B0603551D0F0404030206C030160603551D250101" +
                "FF040C300A06082B0601050507030830200603551D250101FF0416301406082B" +
                "0601050507030106082B06010505070302301D0603551D0E04160414B163E5F7" +
                "BBB872522615BB55850DF3D636E8EA6A300D06092A864886F70D01010B050003" +
                "820201008E5927A04E7D6361819B565EC9C3BFB59F1BBB57007B4534FC5B2F53" +
                "6A02BAFB2EC19DD8EE65DC77F34E2E7A4A2EA1597EE2019CFEBE8BA45BD35A53" +
                "D201465818F1B6E0E891FB380813E27FC1F6D9F3D5DC4C23267FB4326960E7C4" +
                "D69CF837904E694F4DBBD1AA346FC058B158AAC7EDD7F947F67DD09D9183B84E" +
                "B91CDE170895B830F4DE87850BC79C3E27534298BD8C7E7DAF3CC1242777435B" +
                "2A80CBD57C015B1B52A5B46D85187E5FF20EE68A96FB705FAFF4E335EF4FDBFF" +
                "CC5D981943127CE41EFA8C1D7E2E05D03D772E0C02F4443AB3420B6E3D1607BD" +
                "B2416268AB7D9D2B5824AD82B06ECB58B41C0AE092D704F77023F6CC8918F7D4" +
                "3128357B7504276C805E6C27A0A5872C89375F69FF14E91B4A11922DFE7B8142" +
                "8B103B55973214426A063DE2955AAB165CDF105E574F23C00E50FF5B8AB86222" +
                "460734EF5003A0D780DA6FEE9B8CEF0AF45C0BB1D0D142C4A2FDB2BD9776B049" +
                "B70FB4B52D2EF2141E3C3EC8F4BD914209C2F2EB31FAB2311F906EB48D4396D8" +
                "5C5D9B54FDCF52C00FEDB9F399607C1D95BEC1D81BBF8B9E35C36BC72FD90834" +
                "AE8D4A02DFF4FD65DB9748DB626FD4D387A4E0E5ECB73168AF3BB3C788DFAD4D" +
                "CECCE43F9513EA99F1AFFB2B900F5AC55DE8D7AF96B243BA663500A63E4A35D4" +
                "7257229376EE8C0179396C355DFEEEC03F8773BA1DD5B0807E44EA1E11257751" +
                "67020DF9").HexToByteArray();

            internal static readonly byte[] TwoEkuTsaPfx = (
                "30820AC602010330820A8606092A864886F70D010701A0820A7704820A733082" +
                "0A6F3082059006092A864886F70D010701A08205810482057D30820579308205" +
                "75060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E0408C4A9C5FF5EE508D5020207D0048204C8344C1DBD71C360A0" +
                "B4ECAD9F7F43F25272B5C99F1D54A7A0926401082B4E1512610D52B15CBD8203" +
                "36AB72F4E57D083B49BBD606F5C1A268B66E7931E0980A06D8EDF5B5C0BCDA51" +
                "908E21890B24054867912D65125BA0F75B561175A053D2F875C1C846CDC6AFD1" +
                "7599AE877B8CF18585BC405B1E356FD49AAB207BC7C2BBCEF1B0E9FA2EBE205F" +
                "E5F98F825BD9564FA45A7FF011EA41A247AAFA06391C62BAC548A004A139F9C8" +
                "8039B1837066BBF5DD7E8DCEDA3B13ACA5214A53C8D6D748B4DA885CA59741B2" +
                "5799051D59AECE8F06EE0C637406A91070C7DE72B2FAF982BB0A9D937C517D5D" +
                "B0F4D0EB69FDB597F8695A9BC42DD5F87F56119AB5F3E051B0E03E4A069FBF39" +
                "3C5592E1BC28264BDED3ACD7D0CBB5DEE9B426101C6CC5752A5068DF4520C71C" +
                "F10875F1F23BF84D4B6D3A2E133E059A8B1F02B47258F36F84AF4EAC85045489" +
                "971E63970A614CB05C3FA28A711F8DA220C23E463E50B17408E05316F1CB32CA" +
                "37A0C4262081E9C60D897559FF167F2F9B58162399368DD5B85309E9B941FE32" +
                "356258D2EBBBBAB957F496742DB2CF7D8233EBF879887B3F07156BFF9D2CF87B" +
                "D495C684CDF46E451715D4CA1DA21F960868BA70CE88D4D8904EDB97EF69435B" +
                "DC89648A1C330757D51B9D94AF48814109B13EC4AAE6EA99B2ECE5DFDD384F71" +
                "E3A4D39328F5FC55344E2B6EC68D164F57B92AC17D6BF52AF153D431E06770DF" +
                "53A8F14AC579E5E130FC5C3A665E5BEE8CCACC5188191B00EFD13A3A0DD1CBA2" +
                "FAB565CCE5459DCE7CBF8332A3FA1A6E943AB05A2BD28A35025A19DDE18A63E9" +
                "123BBA96B0147221E7CD90CA3A8DFAD634CF8A24EC1AF619CA7B43D844B078CA" +
                "AC708A4D1775AE368B583785307CA3F73C370740DAE2163AADC966BED8EF2648" +
                "28557C1F10BFFED48ACA7ACBB60CA16724F0FD9A2C79B4556A71C7BEAE5549B2" +
                "BDF5B165760DF0C7A0977D923952C9DDC95D89D379E0DF0E292D1534753938DC" +
                "8764451A231F132FE3F40C472CDFED28564002A39ACD4B7059CC284E72D27ED6" +
                "4204DCE2CF30FFF82EE788950CF24358214406CABAB32332CCD7D14A141162BF" +
                "832B1091EEA2C845DE9338D96917065E0CBEFFD292232B20956DFE8116C724F8" +
                "EDB03BED1474460B0C3C45A894C7CEAEA083C2FD5162C926F5DD945BA3BF3A53" +
                "45E82A93F8BCF462AF4C51F4784F8618FC2DB64B4E4A497F0654F573A2F83426" +
                "DFF119440981C9ECBDBA7BDD2AB18F2D62B5EDEA6E2396537EEF3A4264EEC3DA" +
                "4843FF0CD344204C8FDF9C92AF1278937694B30EBE6238AC70D19719D43D77C7" +
                "6B117C4048A7698B822BF371EEC55C1A4C51A13E4A84822F5370616A67B25723" +
                "0BB9B14443A8FAA13414244CFA353E414C9E652C447BE58AEEF982FE4A12FB64" +
                "5FCE47038C15499277FD0EA308036497437DDDF39F596D48FAAC1177112E0929" +
                "234E3F5389DDC21CE14362729AF3EDCD7F2641A8633C13C1E10FBB808E5881D9" +
                "A19778C52E8A8D9DD97766B18EAA9F147AB7B184D7DA131148A70FA0D2FB079F" +
                "E4E4062211D0EF4C3E40D49025BC84C68FC2CAD10F2F5AF80D8174B2A05301D7" +
                "35F3688D854D5D9A2A4646D7F4FD49A16F9432197EB581FB71906AF7D2A0115F" +
                "418AA18C1F14285C7197F3508D374947A8769A91711B0D159A71CA3258529DA0" +
                "C918D8E53E0ADA32E8F32AF11552ED557DC1D8F0F1D027669221C00529B44031" +
                "3174301306092A864886F70D0109153106040401000000305D06092B06010401" +
                "8237110131501E4E004D006900630072006F0073006F0066007400200053006F" +
                "0066007400770061007200650020004B00650079002000530074006F00720061" +
                "00670065002000500072006F00760069006400650072308204D706092A864886" +
                "F70D010706A08204C8308204C4020100308204BD06092A864886F70D01070130" +
                "1C060A2A864886F70D010C0106300E0408E42BF45F0D448170020207D0808204" +
                "90BE13F1716CAB0C61D6269860EF56B1C6695C35AA4B8333CB772579033EC447" +
                "9B83662C6EF04487BD6BD3EFC7BEE8C17F5EAD6E73389A9EFD73FEE959F4FF68" +
                "D31616FA6D24D918F0377555CD68AFFF60F7E7AAF1619C2E1F4B057F5D2CC20E" +
                "DEA3A8683DF2DF5E6D3C062065B38FFC4C16E7AD27BD31742C8732D09114B768" +
                "FFAE7BEFD13303C64E8CA18F2571BBB1FFCC3A28BDBDE510D841FF781D7C615B" +
                "178252E13D24B89D0DE8E47D9160CAE6BDF5E3959BA35218EDC43708F68CB2E6" +
                "B37BFE52D05141B5BBD351C570B18848C68AC15E109467E373904F3AFDA06905" +
                "0C1596D2ACED9D2733EB1FA0CD503B06828944463C579986BBC24B443261F1C2" +
                "2C170F13F3EEAA0FF2EF63200612723AB4A0C768B03C6AC3C6B8D967DA306018" +
                "2D2E3B412EE8E6E0639C282ADEC3899F36D740CF1CEA888824FBCCE2A7A22AF2" +
                "06681597D80B907F50F7044B928BFAFC10F2580B5F7380E2C43BD6F273CD7EC6" +
                "36F3F4F3AD5F2DF9F48CF4B0A4EBB8CB3BA1DDE3448C5ADE45C75CB80CCD61A6" +
                "AFB2E29BA3833A6465C34ACEB7E47CAEDFC1A6B5DB7E3DE594026B648082732C" +
                "1A3804E882DECC2018BCC27A29AAB4B98873099025449B9709EB9C0B5F84EA34" +
                "4A7CE3D0829DE1ED1C89B165A1130DDC8333E54486A94E35A8C017B21DC38D74" +
                "C6C0A685D68103743DD7DFEBBDF0D9BC55DE11DB3F38F762415BA58B2E52CCBC" +
                "72B76D70EA1BA331B24F3817DC635FF59A1247A8F6EB1AAAA257388709BE7E44" +
                "E8A3716A4A2FA0D57E07853EBE9BCE05FDDD23215531BFBAE5E304A9DE44DA93" +
                "6AFF963085CE099665B03357C295D33A338EDFE664EEF200A957484D51736FBA" +
                "73924014152949BDF128248ACEF561F51F08456E03E3533354283A74D0BDA10F" +
                "53BEBD5710C0735C2960188781EBE4B9AC0E658723CBD09C8898FFAE79E5D066" +
                "EBE4586777E1039D38EFF6C74ACB80365341A868C610377C03DE1B691300FC4B" +
                "76A7192FD2D37864CFB336852EE9C2FA8AB96F196A0EDD84876B3278C4200143" +
                "11C933D8C759945E3565C80CE8E7782C4BEC074FA87EF25D951D846FC160A0E8" +
                "ADE67650FAC6B2A7C8F9798CBF8DAD6B23CE7A429995051CFE6941EF7258A10E" +
                "2BD776B6C7DD59823A50BB767E38D6CA099553AD0E3982296DBA8AE6D72CC473" +
                "9F4A9476B113FB06DE80B32F65FEFE0545CB44A3F5940C6776806168C534BC18" +
                "4E73AB8F7B0E855849ABBAFA9D832246329794682331A05DA97848262E9FBCD3" +
                "0A4894B6EEDF42150B23EADB3362C54CEC2CDFA9F36F67AD22389B3487CEDB98" +
                "174644FC2A7C5FEA11488F031AAA30A9F7FE15662C86A0AC5EBAF2FB47CF555D" +
                "B8D1C85410107C2CF40099DA3B281F0DF391E5602B64DA73B585895DCC465338" +
                "29BC0E72F178F41179238049BED59D0602BCAD8AC9E9BC140306D8BDD4150C6C" +
                "4DFD8EA353271B928EA23482E719A58D02515EA83DB54252CB8D466EAC9FD1EE" +
                "71D00DC6D39362883364E31CBC963CD295CDB490074F5C43759D7DD655AA8A46" +
                "EE5FE06E186AB3025E71AAABA9DFD8A105ED86B85FF7C772F596BEEE31CA5BF2" +
                "26BAC1F59123893945BB052A48E9BE2254FF1512B1C2E46D34111F7BA35EA906" +
                "74FBC63FC70A18F095C2AA6CD060A7B52A3037301F300706052B0E03021A0414" +
                "A519F103F114AFAD5EB7F368DB4D0748559CDD190414584DD2F41EC2DBDAEA69" +
                "FB2FF401BD9FC3B57572").HexToByteArray();

            internal static readonly byte[] NonCriticalTsaEkuCert = (
                "308204243082020CA003020102020401020304300D06092A864886F70D01010B" +
                "05003029312730250603550403131E4578706572696D656E74616C2049737375" +
                "696E6720417574686F72697479301E170D3138303131303137323532335A170D" +
                "3138303431313137323532335A302F312D302B060355040313244E6F6E2D4372" +
                "69746963616C20454B552054696D657374616D7020417574686F726974793082" +
                "0122300D06092A864886F70D01010105000382010F003082010A0282010100B1" +
                "74BCC25C16B5B726821C2AD59F0EDB594832A15FE186A087FF40A8BC9396C55B" +
                "C0DB2B3CE3EC5EF26A11AA87073348435417982D29C439FA3EC94A62B4BCC9EB" +
                "CE0236B7E5306B3144E71373B05C24D3C1EE7A4D263BF11FC54D09E4B674F593" +
                "389AAD503930EB7CEFECCA3A64FCCCC15E32E4B304BDAA36267E6BF2DA9584A7" +
                "66E1862758D04E7FF9CC5C46CB074DFBCFAFDC355BF848337CD38331FE8556B9" +
                "F350C6780C7260F73FBCA77FC454247B018E937D2002C9590E07804233EBC28E" +
                "7BC712ACCF6A125EA60B86A87217B23A91866BEAAAE842D0D0D02E87F5F123AB" +
                "811EDCAD7A6819E88B0F0D0932D0748EE02726D7138B1ACEB7A6D4090245DD02" +
                "03010001A34E304C30090603551D1304023000300B0603551D0F0404030206C0" +
                "30130603551D25040C300A06082B06010505070308301D0603551D0E04160414" +
                "610479D21BFEAEC87835A7D03714613D566F25D3300D06092A864886F70D0101" +
                "0B050003820201006511E16CB557E9936363DF92EA22776DD9AD1ED6ECBFE1A6" +
                "B9B718BAF9828AD2B8CE5450F56E1371B54577DE1146FE82089A4331D063DA8B" +
                "3CE786945ABBE4D11F4465068B1C3E4CF987C2C014BAD8BCECCC946EF3C5C56B" +
                "0DDECFB11967A10949A6D111DC96C627D2F1D7969CA6BA7AB131367E89D950C5" +
                "0B756E516CD4CC6BE679BB53684517236846DCE33BB49235CB1601A6657CC154" +
                "8C2D57E5554B2755FD047922B0FAC86437758AB72C1A6EC458DB08D93CFB31D4" +
                "8C723494978D7F32B940E5519852F9B6EA4F0363B59FCA7F8A6CCDE595896B91" +
                "869C1D90FF2CA0CD29F9AA5FF452470B752052BFFBE2CBBE00C68F2DDCBE8C68" +
                "53136118CFD1916D711BF90021C04FB5879BE0974228F79C71532FB2B5DDA1E1" +
                "E89BD5EC9015F2207F7C55D72E976A87479AB62A46465E8D75466492F82CA574" +
                "D788B27A5CA251C9BE3A4EB90F6E7F238CE8AAF46E598DDD70094E148DAE4DAA" +
                "21275F79095ABCE8A47CC362386FDE498D102CCD42B2313AC88FC11E21DF8283" +
                "631329E61F35E8DB12EA8D33DD02B8987C7FC38242412EC97CD52A7A913C6992" +
                "D87E71A75F9654F7F9EDEB80B0BBEA25C5A22CCAF388D109DB0EA7C79247CE4D" +
                "C89F390EB8C6CCC19FA9EB5DFC5BFA4457DB30B27CA73EE1C19934C8BED63E58" +
                "F227222B86010D9521324BDDE04F47BF5190778C6B812ED78AC7DD8C82FBD0C4" +
                "0DA1EE040184500D").HexToByteArray();

            internal static readonly byte[] NonCriticalTsaEkuPfx = (
                "30820AAE02010330820A6E06092A864886F70D010701A0820A5F04820A5B3082" +
                "0A573082059006092A864886F70D010701A08205810482057D30820579308205" +
                "75060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E040830D97AD5E5020804020207D0048204C857D388EF52DCE112" +
                "B6D73E8611F6971BA662ECBA8F80435F83126D213190D6F77369E8D1213CF1C1" +
                "7791AC985C7A0125D1D4728B841FE7BE3E9C44469CBE672377CD97821B49BBA2" +
                "75789D64B648F4F243136E9448166EA366EBBB973041C437E1BA70B609761F03" +
                "03A5ED03C041732D34F070555B006FACA2B8639A0B609D1A7F8A88ABD260CB69" +
                "3365D9181A638A6ADF69385C96BC094E6EEA507160F18A22E0552183C586A45E" +
                "C680CE10B56E1D3956F704C89A0981429C470DB378DE1CF21BD67E9EEB43B37F" +
                "E00117CD3CDB02398FD0D7FD4B48EA38CBF6B19254E957AD2D8A0A4C3AE72355" +
                "590407AC9FE2622C8AF04BF17E62CBB213F9D29999BF184532BC64E2ED1A5323" +
                "1501741A1352F492AFE713503A950DF12E9BB505EBE9C80DF4DB6CC9E1EE0CF0" +
                "02C9CB145E265F7D84A448B9C7462CE25EEBDC9AB3A6E1C996FC535FF7627163" +
                "1CA7E36C1614A9113C96EDD951F2B3B7508DEC6A2BE6889436A741CD170B6201" +
                "D546430CC38CE1E874D78A9D6E4D9665CE8261368FBA08C7456E5B01723D3465" +
                "7AD715328341B4BAD14825D1DB1B0030819B61607E2F4FE76D0EF1616E2C1F96" +
                "4395FFAAA4A9F7E833A1527B630D862DFF5C8DD6EE6557F55B429C9088020D10" +
                "309070D8BD46B1512C0D6B68C8C00EBF215C5DC3DE0BD8B4A92E4C3115687194" +
                "D7DEF524FEA4B02389388C7021BD85EDF13BE19086D08AC682EB8B37F1AC6445" +
                "67CF6213363D889536CD8A4287A9DC16DD5497A8D06A489D6AB12E4943784EDB" +
                "559FC02C7DC1E190A9FBCB8EF7D83AEDB31AE1BA8F356742E539E4A7B9D0A516" +
                "90FCF505BFD5DA6AC8DD67439C2CE9E8D3DFB78A88581BDF0EBB89B810FA7894" +
                "78D5A5BA44BF287BC8D98DF1B286F2B9109430524DFD5739405E46C755F9C943" +
                "03C95FF6E89400D1E1D1E814D795FF0B77ECC84AABFF6A8D3C665770778CE9C9" +
                "A9189DA1E257988AF6588A596F5534D91FA4505581DBB0F8588F97CC3177788D" +
                "131A2F03972DA2753DCEE18965E032A5326CF50378D7D98233A913387315C71E" +
                "3FB2D81A78B537BBBD4408C2E8DA4EFE975EFAA785BDEBA40C5CFF9E25CA07A0" +
                "77DFD9744FE20F783A38A274CDB85A374D1E7723473106DEA578B14C766FEEE8" +
                "6446C61AAFCED190892AE44C8BDC72D5178C3AB1BE9600CA15F5D3383A6219D0" +
                "675F2DBA9AED44BF8702777EA6902344AA572535217EF44BEE37C6E507FEB4FA" +
                "62EF557119608466CD1339C242AFB4F8E0E9EB403E41872B3C5A34B94EF2EBC7" +
                "91111687C764E4E20A25EF07DAE9E402FB08B79EDB4F5B5C3ADAB14E3CA9004E" +
                "FEDECD8DA9D4791BA96F6D6493B301698F2202DB9834B423ACFE71324716FDEF" +
                "6D70FEBC98503E914593A1F511BCD0C39425DAA9981B6BEFA122F8812564D14F" +
                "B6383F3CB8C2C41449E9B58B3D4EE27651C5B20CEBE786312878E641C20531B2" +
                "909BE5727E4C4C01FDEFBC635292A663B53A8EAB29A1FA4CEFF11A02AD511AA7" +
                "F2FD338A86E1876B568074F50B33835186C71C3854945AF082B4DBBD6865581A" +
                "139B3973A3FD5E62AD88C51D636D616AB18EA808BE982C7C51B20FA239D07014" +
                "CEE766F9CDF5D592B1D31881AA5939C670A7CCB48D234268D61031A27D99728A" +
                "4701EF7A241C45A26799C45A8A0A02EA054215973B6F156520544DBE0C3A89DF" +
                "7BBEEB7AB754495781A4A37F4CDD64B1A3A535826B00E1A710AD4D4A56C17662" +
                "3174301306092A864886F70D0109153106040401000000305D06092B06010401" +
                "8237110131501E4E004D006900630072006F0073006F0066007400200053006F" +
                "0066007400770061007200650020004B00650079002000530074006F00720061" +
                "00670065002000500072006F00760069006400650072308204BF06092A864886" +
                "F70D010706A08204B0308204AC020100308204A506092A864886F70D01070130" +
                "1C060A2A864886F70D010C0106300E0408DBF6A30F31A06E2B020207D0808204" +
                "783C0D88298EEF3EDDB8C04416DD28F2DA6654714BD9092529FE1181153A60FC" +
                "552B4D62B0C1F53C6F6337A7C774DBE4396690431E55A963AA509A2351104B91" +
                "E74B9250AA58812954181B1ED602D9699105960C7B82E91362946292E65C99D5" +
                "80DCD3B00FCC0FF4B25095AAF4B5E67886B817556D8B69B3016DE071E31F2B86" +
                "3612A6050FB7D97C5454CE63B842B02FBA72D102DFAFEB01192CCF33BDDEAFB3" +
                "950C53E1452B449950CA860BEF314B32AFABF6B9F2BB1CAFD064960C073239A3" +
                "EEC38BA9B30BDF0A9DBA3FCA6F22F47DC8F593BB7102FE0D8039AE5B2317C9E0" +
                "2DB059C99F06708809362C1676D14D7E5F3DB30E0090697366DDC9900A218E7F" +
                "99851838A111B9C9C9DD9A696DD096DDAA23175164034407463AFA4664BD5E70" +
                "3B6AFD659D6C7CD266ABE731F51F96A2D148B919B83888E4416759169C304A15" +
                "57251FCAE553B5A177DBB5366B031B59149CCE263601A0231544CFD7107BDDFC" +
                "4037AC0AD99F93D001C4CBC4DBCEDE235875C20789BBE8BEDFDE63D1959C25AA" +
                "1E410AB081F03CF5562D814C54A9B66A27E74DC261D4E41513927BADA1E993D5" +
                "20EA81D592B1D4ACEA2577929229B60A8B0AA7499037F3F7F24C1E8E980A06BB" +
                "6B953090844FE068B611DC4C880A4B2FE21F82002C4305A9AE27C1B17607D59C" +
                "89589F122721FFB6DC2D95EFB7D96625EE3C6C252E0E3FF25D4407549358F995" +
                "D9911E9021303FB711B71D7D5F61D6BA845A456B9A832926A098E0EDCA081E53" +
                "9FABBE54C09DF90D37D349FAA3A9259432491307C216E8D3C160E1D5E994161F" +
                "BB29C9BBD789CCEB23591B983D35CE3001D7D4A313A9D66356D5B3BC0BB061D6" +
                "49DFC15CAF2B8B70D3FCF40B1D412E60FFAD5FB4A0F2944F019CD2CD26108345" +
                "20771F437BF27A586AB0866BF1F5920C488648D463A2C430E217CFB080E91930" +
                "05589A9670FA9C75050E45100A553E9A21FE300DE621B12CCA03FE189CE65367" +
                "B1CC452426AD21C67925894905CD594E85C354F85748994C34AAE7E281DC3C71" +
                "81BEE53119708F6C2D29B2CA987F5620650BE2EA087FAF976BB58349E8B67F67" +
                "FAAE11559752C31EB34A72FD4BB23B363255530F92E8C0C82FB7ACD6FF4DD7CB" +
                "AFB831E624522FBDD47A8191957BAF0E9998AB61C5F839B6DFF3A568132A1A21" +
                "643A576D3562EF72294B13E6662ECF9CBFEC602C7AAC5E01D43759917D95BF3D" +
                "D572A45B2CAA118965764470C163FACA61D06693273D869214816431963087DE" +
                "B10FE0316352440E87189532D950E0A9AE5ABE9926907E52F1E4F135B467F4E2" +
                "8BA57B9F371FC934409526E261926E484B465ED10CFDB25A2F3E6838F41C37F7" +
                "DF6A88E7063EBAEB6C426B0A4C7842C58CA49581F14337FDF43FC22DAF79900D" +
                "306E34ED9DACBD6072DF4F34588F28F0F4AE13B431D9F47ACCD6EFB1C4EBBDED" +
                "D507512B820FB02713948D45F3C595330B271308F0E6DAB862BB917EE9E2B1F4" +
                "17CE21A26CC17358C11E2BCC514EB241D0767B4971E9B89F94731843B508C20A" +
                "54345F58E99D430065326FFEA3E1FAC40E24BFB17A51A884CA022944A27436DF" +
                "8F2C4E296A728496C38076FD3F14B007FEFF015EB42329F7453037301F300706" +
                "052B0E03021A0414A0E907FF695A237FAB54BBB94CBCE689EE0B4552041426E2" +
                "132250203B3235FD5023D999B747478D8873").HexToByteArray();

            internal static readonly byte[] TlsClientServerEkuCert = (
                "3082043130820219A003020102020401020304300D06092A864886F70D01010B" +
                "05003029312730250603550403131E4578706572696D656E74616C2049737375" +
                "696E6720417574686F72697479301E170D3138303131303137333331375A170D" +
                "3138303431313137333331375A302F312D302B060355040313244E6F6E2D4372" +
                "69746963616C20454B552054696D657374616D7020417574686F726974793082" +
                "0122300D06092A864886F70D01010105000382010F003082010A0282010100B3" +
                "CBCBEA8EFAAAEDF982CD046188D2F57FE9FE4F4FEA94ADFB7DE47AE731636624" +
                "3C4E5F8C2821BF0B01A32048E8A275FD9B3073E0DA2E8870FD9671BFA600A47C" +
                "8588AAE88150448E6B4C594C5EA288119BE9A3EA369F66EED7C499738B8EAF84" +
                "5E6B36BCEDF7887D061BC86F12D40982199C1C41CCF5440AEF592A482931B541" +
                "1B0E0400FB24AF838BA89A3E1393C24B80B4C67AB923DE68B0B8A2218DA93C2C" +
                "A4908E3B906BF3431839CA6D3FC2A4FC33C4CB58BDEF8982A39DD186D0BB88E8" +
                "E68819C4A7DA4D29F48F1C1F00098DF41C140FA649204564A3AAB9D0E7486252" +
                "77F40B9739ED07D385BF9C0F78528CC5DED08A84B6D392874B2A4EB98B3A5102" +
                "03010001A35B305930090603551D1304023000300B0603551D0F0404030206C0" +
                "30200603551D250101FF0416301406082B0601050507030106082B0601050507" +
                "0302301D0603551D0E04160414743B8DC143DEC364FA69992CB3306A9EDEACEA" +
                "1C300D06092A864886F70D01010B050003820201003A48278ED517B381CA420A" +
                "E3C12C31C22F49D1DC35EA60B018BCB88C6C3E0FF1E7A301E525B5592649B496" +
                "7BA62D89AC748DB026CBD73C9333AAE89C5F5D76FC77E970F214F1508CBFBD9B" +
                "09CD50EF93D61559004027E80BCE6809F196633C22A13CA2CD7596ADE5B4A23E" +
                "BB4E03367892433357636316F0D235D3DACFEB99631CB437DEB04E4A074B1CBA" +
                "6C6D722210B058285A3E124EC8148E31B3DE37DCBAECF560076FC0E50965D482" +
                "DCBF529127CBE2440BA746DC71765516D480E68805C7668A0B538BC489DA2FE5" +
                "E158BB6786A626BF0AAB74AF042347785B54323CC019CA84347BFF57C025791D" +
                "69C558A605D46C50297DE1E9576972053A3DDFE5EC8FD2DE0D48B80252C39EE5" +
                "4410AD46D8128A453758DA442CC2879290A50232B13920D9C6800D8773C2FD82" +
                "D11755C336CD6416FFE26F97599A29E5D18227949DD4385C74D29547D6C70ECB" +
                "CBE006AE2D18BCB8708C50E7C46D442A8DEDECEEDCDEAEC47042957B1D18D410" +
                "96350438DCD8223B17E5FDC3C9C0D9BD47D191E6142A601930817E30F8E0509F" +
                "D5F5FE269683494F08C55B9ECE6E3D503C81A81F40CC959B858E52BA086D95B5" +
                "8DC13492C128B64F862E7800384C98A1303EA570D328C003E2B944A804B9794F" +
                "A5C6040881E61510E90C20F21DEA0A73E0DA5C1A2D178A48D76CC8FAA2ADA660" +
                "2A98B50AC197B40348A012E83A98EFF2B8D64900DE").HexToByteArray();

            internal static readonly byte[] TlsClientServerEkuPfx = (
                "30820AB602010330820A7606092A864886F70D010701A0820A6704820A633082" +
                "0A5F3082059006092A864886F70D010701A08205810482057D30820579308205" +
                "75060B2A864886F70D010C0A0102A08204EE308204EA301C060A2A864886F70D" +
                "010C0103300E04084BEC68D6BA25EEEE020207D0048204C8E534D459438245AC" +
                "EAC40458DAE1FDC42FB28605913F53F859CA1B7D62E385E48352F956A1B4D904" +
                "B796CE577F18A5E334990367542F7EB0806EDE06892812F914D62BC029E6637D" +
                "60BB10F125350DFC7F7702D68984F79C9192ECC07178330141B2CD8D88974F30" +
                "DD2CDF6F82F668AA5BF9F3201F1A8ED58B2546A2683260751F8254C3AC574ED2" +
                "8FBEF421A4C8B1C2CFB4691875314C06148A801036E39827BF3AA57F9FC32DB0" +
                "00C0BB9CCEB0E828651AD82E903B710DE378B00533994AEA596AD4FFD5B075E3" +
                "4BA54099F8FEA4AFCC469D071C48A0EF8BB58B46D3666A251188209AB7FE80F2" +
                "238EC6977280497ED7833283D3C49DD9546190404E1018C1179DDABBAF18D9B9" +
                "FE18D71FA4A87976F95A533FAE01E96057CDB05FF19DD14673AABB7FB5C3B01A" +
                "44F7D8265B320E846244E7C3EE65D01F4B468084F3890D92A065D745F41275D2" +
                "9650B00BF3CB7B4DB64D14A78591147F9FEA71DE4469DE58D1E90A40B5ABB151" +
                "B4F24BCA90E1966C9588D96627FE1F69ED8BC7A52F03BEED75CA8E90EBB811B1" +
                "37BABB34129C5E7AE44E1B6FE3A3DE8EC923F05E4A471BA0D27B134E5880CC4B" +
                "20CE7404CE9EB2C114C7018E811786A7FFF5FE6A2C7FAC4C4B2FA0CA6223E9D4" +
                "0A9D6567FB659857A83703D9E995CA2E1BC96FD6EAC678204661CD866530E61B" +
                "40533A011A250B6632760FD942A8A5741410C1BF0212D66085BB623F5C53A186" +
                "6699CE7CAD843C9325D54D260D254B3273717DEBE43F4F7FEDFC984546434CEC" +
                "46D70E3B888A85A11252DC12E7D50A32CAD5D4544C161F81BBCEF0D0DE893F25" +
                "C762FDCCDE1DF91262C815C925BFC6BA133E5CD42D32E7D2A6FB0BF22AE8482C" +
                "CEAE15070F1692D5BF3C2E2CCC02D77DF879C4D4F188B80870A234714B92197C" +
                "7A27F39A7E5366D7A2E99BCBF8BC5576DD627754EC4AEE2DE118EEDF7686D109" +
                "0B3A59E97051624D56224423BC9B4A2D7A85549596D4F981E29BF4C7F6CA8F38" +
                "2E4138BF515BB72CF4CF5D44F49274843BCCED64A9CA5A514E29C2DFE75CA4B2" +
                "FABACCA238DBA202AC6A996BDE4F79C9A568450C849BF12063C0CCEFF4B81057" +
                "A386697D217F4AD02ADCA127271CC5A3E5E9E3F6722D4173B83B39CEF5DF0BF8" +
                "F9630D575F3A99BF83BDA8A5A1A8CA5AF876F14BA5360DC6CBA5D1EC2F46C3E2" +
                "F14F87A03052C3EE4994C6B661248401F40EA0843F011E5186EF09B8917B4218" +
                "DF0352289252463E2DE1FFED603F330D80D6349C36FA9CF9B721069BDD833491" +
                "52FB1BA9F994507BB22CA076AE781FCDC5B1A487C9EE8BC2C83476260F61D866" +
                "0981A7BA1F5471F56C067BC6A3C1C6F4F76C72497DDD8AB961DB8FF673B00EB8" +
                "7C498624B2C46A184B6D4F248AC9307DF046DCFC70811402AF06468E7E07D6FC" +
                "353830F4F06B5902FD823A4A94099AAD9CEEC531CC544F74BCE62777F188CAC1" +
                "B819F6F0F449372E1B3ADC45514C8598A18427D2957511938CAC5439768A97C4" +
                "8D05A62660628DC2C16D1CFE73920C00B6DBEB4D66D1572F60817FBCCC3DDEB4" +
                "C5DCBB69799E4D3BC155FCF564B6AEBF25C7AC3A0F0CE7F4F2761738A7485236" +
                "8B4D950EA6672BA4615A9A040FA5166FF520948CA1D3649D9AD2317B8380FE90" +
                "4644F2C06F6173C8FB52A572FC50D69C273DBB63EBACA717441C2530CDEDAC0B" +
                "3796FAA4F9BAEC808A01F70E5B48E42B2AA49AFB65054EEFCB0F10072A38DA5E" +
                "3174301306092A864886F70D0109153106040401000000305D06092B06010401" +
                "8237110131501E4E004D006900630072006F0073006F0066007400200053006F" +
                "0066007400770061007200650020004B00650079002000530074006F00720061" +
                "00670065002000500072006F00760069006400650072308204C706092A864886" +
                "F70D010706A08204B8308204B4020100308204AD06092A864886F70D01070130" +
                "1C060A2A864886F70D010C0106300E0408FDE1F090274C2210020207D0808204" +
                "8071C27010C2CFFD78D49D28DBB5AEF4269926F1055E9DF0471FFD7077366628" +
                "DB87AC150E12BD5C636BE9A9CCAB194C87F32749D9544873A0C2360453A5293D" +
                "E03D5DCC1E3F12FDD89CF5BC37099A08DD9700CC0B6D1E351D5451386513E774" +
                "C3FAAB375F7A6731C8923C7979F8B283D33C09CF1DFAEFA9EC014845CD779C24" +
                "A1B085C2F9A8B3AF35197ECAD885B926463D785E9C192326FC6B956695970920" +
                "8B571E383D1083AD446DFEFADA128E247BC038DC74C43E9B5832922814DCEAED" +
                "F97E989BDB105642466FE2B2E2FEC3A19F556766DBBD0BCF9C5627876B1D2A19" +
                "42B3EC2DFA957DE74E669AC767811C6CE6133ED50F97D29FA002FAA0A72980A5" +
                "CC66C5FB86B1F657BE940489087D473AE6A5475A952B24A585254FC27AD50D80" +
                "0C3A77B13AC17401453645C03FFC66984FEA1ECC385852EFD7EB008971E31652" +
                "43A3F672FFB8590F756B4481EB5FFBBF5935C8062741D7AB38A4BE28D141FF16" +
                "6773ADE4EC3C021DDB5FCC43E2D03CDA4AD6BAC0EE63CA816089B8C971A3244C" +
                "6186BBEE09C09A31BC29DC8EAC665E7D9C6CC148C3FAD9028FF2E0A9F727DE1B" +
                "08AD8561D918677FB026A8EBD9BCFB0E106C058C6EFFD53FE13996B9C40F12B3" +
                "90DDD3CC63107CCA59F1CF861692629F4AFD7D3257218D9F888DF2E7A67BD601" +
                "2EEF48B82D09F01D1D1EDCEBEE878A771399D58EEA1D50C8F8ADCA3432696F2D" +
                "49D7253F064F4EC28C335853CD4EF9E72094517EFE61EB3E0A7F50451C1F9CA6" +
                "B95FCF62F36E6D79B16D99119C9E0D4AFFB36972E68A1FA2E2A099596176EA5D" +
                "C09756632BB4FC82136BB554FB32A2997DD982BDBD058F9990403FF786342C91" +
                "F117A98F0076CAB6AAF35F8FD5A5E795F9F3368049849D582E9950B42924BBC6" +
                "5EA3A0FB8FE87FD0B74C1621B52B673E92FD6475DCCB138425740AD1C006ECE5" +
                "3633C9AA2B02404384A31F7FE7DC9F41F2CDB4584283B48E6C5D1EE659316F2C" +
                "FAB3E6D5AEE5F0F986D223E7077C417BD890035E373BAC6B90E62451B14BB701" +
                "30C263872865F8975165452074A53FD036FC3930FF2F781D2805999DA7955BB6" +
                "4D4A98D0E67E4C18F73126177D047E89B24A409DEEA7B7B8CAABD0DD39DF3818" +
                "59E973360EFEE67D8FF9E24275E9F47D5A67E812DD68F8325603F3040180D85E" +
                "2EF61BD821CA666550E55F8A8C172D93315C6D7E14567BD9DE84A0C298A6A586" +
                "A63611495C66DD7D9AF83E20CEF31F3A3D3CC25553F2045FBB547C97C1AE43E1" +
                "F0147C116C82AB442562534E0898023CFB459DB66B7320DECA41E26477D76D5A" +
                "562BAC06023B778271311E3BC8E2D5ABD90515650A994CDAA99D38F6169D480A" +
                "122AD01731819FEEB8FE3111C1CF06FB9B5882EB74232B0CDA7DCAD2988FA9F1" +
                "8F4DCDC7DDC45983128E07F6687AD6C1FB219B3295AA74592B3D8B05C7D9851D" +
                "140A71CAD10D18EBCCDEB9C9243B39772E29D923D9D5014EA4CED9FE5C7859BD" +
                "6AAE8902726D5ECBB6C0D028728F316034F991CEA8F435FFDF35B26583C1CC44" +
                "959BAE4514EB478327BA76E6F9B759E176FDD8BFEDA8BDE6E7BF5E2F9C76CAB8" +
                "055DBE8A531FA77690A620BC5463431899B7577E3AEF0D382F31955AB5FCC951" +
                "973037301F300706052B0E03021A0414F2009473DB4B8B63778E28E68613246F" +
                "D53995810414A911A22D051F1BF45E7FF974C4285F6B4D880B08").HexToByteArray();

            internal static readonly byte[] NegativeSerialNumberCert = (
                "308202C2308201AAA0030201020210FD319CB1514B06AF49E00522277E43C830" +
                "0D06092A864886F70D01010B05003014311230100603550403130953656C6620" +
                "54657374301E170D3138303531343231303434385A170D313830363133323130" +
                "3434395A3014311230100603550403130953656C66205465737430820122300D" +
                "06092A864886F70D01010105000382010F003082010A0282010100E2D9B5C874" +
                "F206A73C1FC3741C4C3669953929305CF057FF980DE2BAAEBA5CF76D34DE5BF8" +
                "ED865B087BF935E31816B8B0DEDB61ABEF78D5CBC5C3389DBD815ECF7BDDC042" +
                "BD21F68D9A7315671686613260B5D3505BC7C241C37C6E42C581228965B5D052" +
                "B8ED3756966C6B678CDF647E5B659E5059915B16EF192BD313F6423FD487BBB4" +
                "06BA70A7EDA7DBC55E15F1D93017BC92238D22DD9A15176A21F1BF7CF7AD280F" +
                "7C32D76C8B136F0FBD0A89A373CF194CB5A0CF7AC1FA8098458290FD66311BB8" +
                "DB17E9CB70D6668A8926F26198586C796C61EEDBFD376525130BAD2C452B6A66" +
                "97AF7FE8F91408785C20DE79F4AD702197734328B4642FB898FF730203010001" +
                "A310300E300C0603551D130101FF04023000300D06092A864886F70D01010B05" +
                "00038201010057E4CE63680B165CD3DDAF5CC84F31CBC6C2CCDCBE33893A3FA7" +
                "01781BBB9F003C0E301216F37D4E9C735F138920121983728EA88324DBEC66C7" +
                "824BB25EC534C3AA5999C8A6D9393F9B04B599A94B415D0C0392E0FBE875221D" +
                "A4348FE76ACBE5D845292AC475A9104052CCEF89390118E8EA8051519F423276" +
                "164D8BDD5F46A8888C540DF5DD45A021593A0174BB5331223931F7756C5906A1" +
                "94E7FF2EFB91D4CBFA4D1D762AE0769A07F2CC424692DB37131B2BEBDB3EE4BE" +
                "07F9B1B12B5D368EC9317237CAB04D5B059F9226C972BADBB23CA747A46581D4" +
                "9240E6FC0F2C8933F1844AD5F54E504DDAA3E87AE843C298DED1761035D2DFBF" +
                "61E1B8F20930").HexToByteArray();

            internal static readonly byte[] NegativeSerialNumberPfx = (
                "308209C40201033082098406092A864886F70D010701A0820975048209713082" +
                "096D3082060E06092A864886F70D010701A08205FF048205FB308205F7308205" +
                "F3060B2A864886F70D010C0A0102A08204FE308204FA301C060A2A864886F70D" +
                "010C0103300E0408F693A2B96616B043020207D0048204D8BEC6839AED85FA03" +
                "2EAF9DED74CE6456E4DB561D9A30B8C343AF36CEA3C7F2FA55B1B7C30DBF4209" +
                "58FEB61153386DCE8B81DCC756E06EC872E8B63FD6D8341123FDAE522E955357" +
                "76B5A09941595C79404E68729389819D05459515314167A9E2585F3B2C8F9F24" +
                "860BA42612DC492137051063A8EC8CBBC3D0ED59B92D10E6612C0C9AD5AE74DD" +
                "999007CFEDA7A1AD2A2D5C25E41201F5A810D2654A3DA1648AF5D9EBBCFDFEEB" +
                "FF8BC78A7BCE26EF0ECB1B0F07B33FB777160ACDF3DE00267398D771A35740AF" +
                "661B189CB7394C02599417AF601231F3B6F40869E2DA8909D8A6D9565CFA389C" +
                "5A002193B3CC374D973F0A587697CE6812126E3BC390093E56B671E8FA77E1C1" +
                "E56AE194AD7695BED88A1139AA149F4E0875121995661B4133E9242FCAAF8A1F" +
                "5721A055FC523EFEA4EEA98B6FB48EF07891B2E80D6EAC824FE9BBFBCB6A72A9" +
                "52F12C3E3DE63F33F9A126BAC8FB0C7F2BF98A781598E6FE5A86409F4F22249F" +
                "4A38B4670A2BF0EF93B0990AF33672CCAEFF15BB5813ECEDFDA38CA2BEEDAAA5" +
                "2B880609057B3FC728F6D1964B7185A933B1526666FBC42C1B5D9A6FF3E2B3BD" +
                "1BB8893BB6B9170BD4D7C6EB47C573FA90E19971017A6FAAD2B36CC746EBB238" +
                "2211248404A09E2ABBC546D85B212286334E164FE24E03BAE053C3C12FA21F29" +
                "FAC8A69F0EEBC688FB50834B5D7F2CB7CECC9BD5D6C06200AAC11B33AF2B6E11" +
                "1F3853AEFBC086881561860093AE6C66A488BECE95CC80ECCABCFE91BDD302EC" +
                "B86DF4B99530ECD707943E76ECA51E07DC109B9B788D1C0664C560352B251FCF" +
                "D9F57C6C18DEDFB9B848A2A57A96B0BEB6C20FEFADAAE468D96B48AB061F6BE6" +
                "194865E676009BD48E5951C534C05A4C4186DF7CF85B596E2F4FA2495440071B" +
                "C13ECE7AF0E58FA13B53DF115BBAF846A5E2AF1F71B3F8AE895C62ABD90FBEBB" +
                "815993D122B4B6816DF95C8097E9A1FC66F0181441B3BC3320BBABEE02004138" +
                "93BBFEB480123A28B14AD4D4B8C918B851DA11C4347C8D1E46B2F211F0E6F049" +
                "0674C2B1C2F6AC020D79E50A45363AD54D5CD2D6668DE917111111E380ACC015" +
                "3CF5A0E3B25A4DF35305308FA2A5C0FFFFA31F6699D5F0699DD82AA5502EA6C3" +
                "B6B782FDEDF5515DBA4D1FDB295A77119D49BC098D35A45E8DB2C88B7C97DD1A" +
                "BB54B0D2EA27AD7B7A79D3C707C482376F8F612129C48024F4A1065BFCFAC2FE" +
                "A176AAA2E0064BE2E77D9E3BCEA0AA732B59DB213A9A0EC972E376D8ACDE5BB6" +
                "4C668B0E24BFE9479BE7574B07D4BEADF821769720A0B7557411826C97206B3D" +
                "176262F4BBF64631ECBD31418487588773E39D46784A4361E31742333BE1DE2A" +
                "BB0331011CA400795E306EDA88F82C78B6906B22E5D0CF68F9F349B9725178BA" +
                "6B03D972E27C8EB376BE51206DE070FA45EE430D6CE17FD2744A7830E7A8E2C4" +
                "C5033D39BFB14D74214ADE082371C49246540B0593487B0DC95BC9B887FA2222" +
                "250D59EB2BB04983AD59004E3921ADCADB30E5C713B9B5FC31A6FD97D6694483" +
                "2E29DAD105CF81265F8149B5EB32192CD0FA45F53621AE7A4D372ECCEC013D08" +
                "B7D1BE73E5D3331236E90DE00A3AC3044F1A5462C9E84CB621A9C68C8B954824" +
                "B684ED3DFC2D2839B19AF443F54ECBF3EC580BADC3CBECFDFE05BDCBA201C984" +
                "C7DDE01EF79E57C5E8EAF9C1E4C8D48DCDA8C6A51F8C0DECABFC4BA6B776C4C8" +
                "BA0110C6D5CEEBC7476E86D4710F5E773181E1301306092A864886F70D010915" +
                "3106040401000000304F06092A864886F70D01091431421E4000380038003100" +
                "3800350063003300300061006100660038003400370036003500610037003700" +
                "32006600610036003400350035003900630062003400320064307906092B0601" +
                "040182371101316C1E6A004D006900630072006F0073006F0066007400200045" +
                "006E00680061006E006300650064002000520053004100200061006E00640020" +
                "004100450053002000430072007900700074006F006700720061007000680069" +
                "0063002000500072006F007600690064006500723082035706092A864886F70D" +
                "010706A0820348308203440201003082033D06092A864886F70D010701301C06" +
                "0A2A864886F70D010C0106300E0408EBDF03E906A5A3C1020207D08082031043" +
                "B764B7A7FA6203394F0625BB0D59B39D3F28197972CBEC961476D76BE560A312" +
                "1ED025A9455F39C33B2ABC1FA897EC29FAF01970321069F8CA20EE4DF6F9E98B" +
                "0DCB3A290DFF4B9B1D151E4AE3AFD7D7383B40B292B0F369A47823A7C175F6CE" +
                "A327283B8B33712B88F06ADDA6CD626036765E9E102E110C8E44EA75E53C5FEB" +
                "661B08D7A9C06B59A35E76F293E8EFA64AF425B877D0321C13BA0079DA14D3A9" +
                "A76EB69CF22E4D427A35C7391B650B822FA91FF0D3DFD1E037C34E2F1110C1A7" +
                "F88A35E560422AE31203ED22BA74D1A6F8FB2AB5DA6E5E66633B7081E6F5681E" +
                "DE01BD28E6DA6B6E0C1444A010DE229D6AD6E44E99400EC22EF1DD6E7E805503" +
                "4B1A29ACD78F6D561E2DA35B84DA1F08BCDD0A74AF49E509744A91F4304F7EEB" +
                "6231C18EC9159D31114183C0B9621AA227663AC67A5DB4B0982B539578DF0ED7" +
                "6B577FD370FA4330E2DFAA015E1E6E2E4C6B71967071390D4B1CD061F567D072" +
                "2BBBE027C1350B1679D1AE7DE7522AF62960403C15F1B75DC2AA0A1339051FA9" +
                "E16692773046ABE0EEC29C1E6FE0ADEA33F7F878B004BC161D362ABDCD2F9992" +
                "82D967B9FA46BCBF0D12CB6659FB2BABA258DE158F51F961CAD7AB5D68D6DFE8" +
                "5E0DC6F043C4AB203E2F549EE0A3C2E25E13CD1462CD03AA3DB4689BA1DD0D8E" +
                "D8293DA3D195901405154E98999E60BBA39A543E64963BE607BD48275508946B" +
                "D27DA136C9F53C692DDC81FD3F6AD56419589716EF6F3A66A35049B29D438D5C" +
                "F778961F74E954CBE1395D8A98971C30C3941CFCF2D3B8851C819D79D164FC71" +
                "CA414798F4FFF3A6362A59AA17F543B5B4B2F2B7E14ED48B8ABEEB68469161B0" +
                "6D2233257291C6F1B67D0BDC2422F3E04213CA29EB648219A6C3E339C1352E55" +
                "F6D8749592723934693DD54F1D098DACFE9D308556B060CCC75D2F7AA8DDEC3C" +
                "B1B127DE82AC8489FEFA4A9F09D49C919F703236036853D5E802975D4B3DA619" +
                "EF90CF53AA38D0D646B69E75751DA833C619337722CA785477343614ACB67AF4" +
                "427E3EAFAC5900F0A573512BD81F1A86A848321309156093665E39193B0867A0" +
                "5C86500AF2760F8A87DB8E6E5FA2753037301F300706052B0E03021A0414AA38" +
                "B6B4257EBA3E399CA9EDD87BDC1F82FBF7D70414B796F49E612D1A2B5A0DEC11" +
                "608A153B5BCD4FE9").HexToByteArray();
        }
    }
}

