-- -*- coding: utf-8 -*-
newPackage(
"TSpreadIdeals",
Version => "1.0",
Date => "February 01, 2021",
Authors => {{Name => "Luca Amata", Email => "lamata@unime.it", HomePage => "http://mat521.unime.it/amata"}
           },
Headline => "A Macaulay2 package to deal with t-spread ideals of a polynomial ring",
DebuggingMode => false
)

export{
"tNextMon", "tLastMon", "tLexSeg", "isTLexSeg", "tLexMon", "countTLexMon", "tVeroneseSet", "tVeroneseIdeal", "tPascalIdeal", "tSpreadList", "tSpreadIdeal", "isTSpread", "tShadow", "fTVector", "isFTVector", "tLexIdeal", "isTLexIdeal", "tStronglyStableSeg", "isTStronglyStableSeg", "tStronglyStableMon", "countTStronglyStableMon", "tStronglyStableIdeal", "isTStronglyStableIdeal", "tExtremalBettiCorners", "tExtremalBettiMonomials",
--service
"tMacaulayExpansion", "solveBinomialExpansion", "initialDegree", "initialIdeal", "minimalBettiNumbers", "minimalBettiNumbersIdeal",
--option
"FixedMax", "MaxInd", "Shift"
}

-------------------------------------------------------------------------------------------
--  Compute the next monomial (optionally with max index fixed) respect to mon
-------------------------------------------------------------------------------------------
tNextMon = method(TypicalValue=>RingElement, Options=>{FixedMax=>false})
tNextMon (RingElement,ZZ) := opts -> (mon,t) -> (
if isTSpread(mon,t) then (
    S:=ring mon;
    n:=numgens S;
    oldmon:=support mon;
    mu:=max(index\oldmon);
    j:=#oldmon-1;
    g:=n-1;
    if opts.FixedMax then g=mu;

    while ((index oldmon#j)+1>g) do (
        g=g-t;
        j=j-1;
        if j<0 then error "no t-spread monomial exists";
    );
    newmon:=take(oldmon,j);
    g=(index oldmon#j)+1;
    k:=#oldmon-j;

    if opts.FixedMax then (
        k=#oldmon-j-1;
        newmon=newmon|{S_mu};
    );

    while k>0 do (
        if g<n then newmon=newmon|{S_g} else error "no t-spread monomial exists";
        k=k-1;
        g=g+t;
    );
) else error "expected a t-spread monomial";
product newmon
)

-------------------------------------------------------------------------------------------
--  Compute the last monomial (optionally with max index fixed) of the shadow of mon
-------------------------------------------------------------------------------------------
tLastMon = method(TypicalValue=>RingElement, Options=>{MaxInd=>-1})
tLastMon (RingElement,ZZ,ZZ) := opts -> (mon,gap,t) -> (
if isTSpread(mon,t) then (
    S:=ring mon;
    n:=numgens S;
    oldmon:=support mon;
    mu:=n-1;
    if opts.MaxInd>-1 then mu=opts.MaxInd-1;
    g:=mu;

    ind:=#select(index\oldmon,x->x<=mu);
    j:=0;
    while mu-index oldmon#(ind-1)<t*(gap+j) do (
        ind=ind-1;
        if ind<0 then error "no t-spread monomial exists";
        j=j+1;
    );

    oldmon=take(oldmon,ind)|{S_g};
    k:=gap+#(support mon)-ind-1;
    while k>0 do (
        if not isSubset({g},index\oldmon) then (
            if g>-1 then (
                oldmon=oldmon|{S_g};
            ) else error "no t-spread monomial exists";
            k=k-1;
        );
        g=g-t;
    );
) else error "expected a t-spread monomial";
product oldmon
)

------------------------------------------------------------------------
-- return the t-spread segment of S between ini and fin
------------------------------------------------------------------------
tLexSeg = method(TypicalValue => List)
tLexSeg(RingElement,RingElement,ZZ) := List => (ini,fin,t) -> (
S:=ring ini;
R:=newRing(S,MonomialOrder=>GLex);
RM:=map(R,S);
IRM:=map(S,R);
n:=numgens R;
l:={};
mon:=RM ini;

if isTSpread(ini,t) and isTSpread(fin,t) and (degree ini)#0 == (degree fin)#0 and ring fin===S then (
    if mon>=RM fin then (
        l={mon}|while mon!=RM fin list mon=tNextMon(mon,t);
    ) else error "expected the first monomial greater than the second one with respect to lex order";
) else error "expected t-spread monomials of the same degree";
IRM\l
)

---------------------------------------------------------------------
-- whether a t-spread set is a t-spread segment
---------------------------------------------------------------------
isTLexSeg = method(TypicalValue=>Boolean)
isTLexSeg (List,ZZ) := (l,t) -> (
S:=ring l#0;
SL:=newRing(S, MonomialOrder=>Lex);
RM:=map(SL,S);
l=RM\l;
deg:=sort flatten degree l#0;

if unique(l/(x->isTSpread(x,t)))=={true} then (
    if tLexSeg(max l, min l,t)!= rsort l then return false;
) else error "expected t-spread monomials";
true
)

-------------------------------------------------------------------------------------------
--  Compute the Borel monomials generated by mon
-------------------------------------------------------------------------------------------
tLexMon = method(TypicalValue=>List)
tLexMon (RingElement,ZZ) := (mon,t) -> (
S:=ring mon;
l:={};
if isTSpread(mon,t) then (
    ini:=product for i to ((degree mon)#0-1) list S_(i*t);
    l=tLexSeg(ini,mon,t);
) else error "expected a t-spread monomial";
l
)

-------------------------------------------------------------------------------------------
-- count the number of monomials of the initial lex segment with assigned minimum
----------------------------------------------------------------------------------------------
countTLexMon = method(TypicalValue=>ZZ, Options=>{FixedMax=>false})
countTLexMon (RingElement,ZZ) := opts -> (mon,t) -> (
if isTSpread(mon,t) then (
    n:=numgens ring mon;
    supp:=index \ support mon;
    d:=#supp;
    dd:=0;
    if opts.FixedMax then (
        n=max supp+1;
        dd=1;
    );

    l:=for i from 1 to n-(d-1)*t list {n-(d-1)*(t-1)-i-dd,d-1-dd};
    c:=0;
    sub:=0;

    for j to d-1-dd do (
        s:=0;
        if j>0 then sub=supp#(j-1)+t;
        while s < supp#j-sub do (
            c=c+binomial(l#s#0,l#s#1);
            s=s+1;
        );
        l=for i to l#s#0-l#s#1 list {l#s#0-i-1,l#s#1-1};
    );
) else error "expected a t-spread monomial";
c+1
)

------------------------------------------------------------------------
-- return all the t-spread monomials of S of degree d
------------------------------------------------------------------------
tVeroneseSet = method(TypicalValue => List)
tVeroneseSet(Ring,ZZ,ZZ) := List => (S,d,t) -> (
n:=numgens S-1;
l:={};
if 1+t*(d-1)<=n+1 then (
    ini:=product for i to (d-1) list S_(i*t);
    fin:=product for i to (d-1) list S_(n-i*t);

    if ini!=1 and fin!=1 then
        l=tLexSeg(ini,fin,t)
    else l={};
);
l
)

------------------------------------------------------------------------
-- return the t-spread Veronese ideal of S of degree d
------------------------------------------------------------------------
tVeroneseIdeal = method(TypicalValue => Ideal)
tVeroneseIdeal(Ring,ZZ,ZZ) := Ideal => (S,d,t) -> (
l:=tVeroneseSet(S,d,t);
if l=={} then l={0_S};
ideal l
)
------------------------------------------------------------------------
-- return tha t-spread Pascal Ideal
------------------------------------------------------------------------
tPascalIdeal = method(TypicalValue => Ideal)
tPascalIdeal(Ring,ZZ) := Ideal => (S,t) -> (
if t>numgens S then error "expected value of t less than the number of variables";
ideal for j to t-1 list product((k->S_k)\select(numgens S,i->i%t==j))
)

------------------------------------------------------------------------
-- return all the t-spread monomials of a list
------------------------------------------------------------------------
tSpreadList = method(TypicalValue => List)
tSpreadList(List,ZZ) := List => (l,t) -> (
if #l>0 then (
    S:=ring l#0;
    n:=numgens S;
    lt:=flatten for i to n-t+1 list for j from i to min(i+t-1,n-1) list product {S_i,S_j};
    K:=ideal lt;
    l=select(l/(x->x%K),y->y!=0);
);
l
)

------------------------------------------------------------------------
-- return all the t-spread generators of a monomial ideal
------------------------------------------------------------------------
tSpreadIdeal = method(TypicalValue => Ideal)
tSpreadIdeal(Ideal,ZZ) := Ideal => (I,t) -> ideal tSpreadList(flatten entries compress gens I,t)


------------------------------------------------------------------------
-- whether a monomial is a t-spread monomial of S
------------------------------------------------------------------------
isTSpread = method(TypicalValue => Boolean)
isTSpread(RingElement,ZZ) := Boolean => (mon,t) -> (
sup:=index\ support mon;
if (degree mon)#0 != #sup or #(flatten entries monomials mon)!=1 then return false;
sup=for i to #sup-2 list sup#(i+1)-sup#i;
if min sup < t then return false;
true
)

------------------------------------------------------------------------
-- wheter a list of monomials is a t-spread
------------------------------------------------------------------------
isTSpread(List,ZZ) := Boolean => (l,t) -> (
if isSubset({false},l/(x->isTSpread(x,t))) then return false;
true
)

------------------------------------------------------------------------
-- wheter an ideal is a t-spread ideal of S
------------------------------------------------------------------------
isTSpread(Ideal,ZZ) := Boolean => (I,t) -> (
S:=ring I;
if isMonomialIdeal I then (      -- or I==ideal(0_S)
    gen:=flatten entries mingens I;
    if isSubset({false},gen/(x->isTSpread(x,t))) then return false;
) else error "expected a monomial ideal";
true
)


------------------------------------------------------------------------
-- return the t-shadow of a t-spread monomial
------------------------------------------------------------------------
tShadow = method(TypicalValue=>List)
tShadow (RingElement,ZZ) := (mon,t) -> (
S:=ring mon;
n:=numgens S;
l:={};

if isTSpread(mon,t) then (
    supp:=index\support mon;
    supp={0}|flatten(supp/(i->{i-t,i+t}))|{n-1};
    ind:=0;
    l=flatten while ind<#supp list (
        for i from supp#ind to supp#(ind+1) list mon*S_i
    ) do ind=ind+2;
);
l
)

------------------------------------------------------------------------
-- return the t-shadow of a set of t-sprea monomials
------------------------------------------------------------------------
tShadow (List,ZZ) := (l,t) -> (
if l=={} then return {};
S:=ring l#0;
n:=numgens S;
SL:=newRing(S, MonomialOrder=>{GLex});
RM:=map(SL,S);
IRM:=map(S,SL);
l=RM\l;
IRM\(rsort unique flatten apply(l,x->tShadow(x,t)))
)

------------------------------------------------------------------------
-- return the ft-vector of a t-spread ideal
------------------------------------------------------------------------
fTVector = method(TypicalValue=>List)
fTVector (Ideal,ZZ) := (I,t) -> (
S:=ring I;
n:=numgens S;
f:={};

if isTSpread(I,t) then (
    indeg:=initialDegree I;
    f=for i to indeg-1 list binomial(n-(t-1)*(i-1),i);
    sh:={};
    for i from indeg to (n+t-1)//t do (
        sh=join(tShadow(sh,t),select(flatten entries mingens I,x->(degree x)#0==i));
        f=join(f,{binomial(n-(t-1)*(i-1),i)-#sh});
    );
) else error "expected a t-spread ideal";
f
)

------------------------------------------------------------------------
-- Compute the i-th Macaulay expansion of a (Shift=>false)
-- Compute the i-th shifted Macaulay expansion of a (Shift=>true)
------------------------------------------------------------------------
tMacaulayExpansion = method(TypicalValue=>List, Options=>{Shift=>false})
tMacaulayExpansion(ZZ,ZZ,ZZ,ZZ) := opts -> (a,n,d,t) -> (
macexp:={};
as:={};
dw:=d;
up:=dw;
if opts.Shift and a==0 then macexp={{0,2}};

while a>0 and d>0 do (
    while binomial(up,dw)<a do up=up+1;
    if binomial(up,dw)!=a then up=up-1;
    macexp=macexp|{{up,dw}};
    as={up}|as;
    a=a-binomial(up,dw);
    dw=dw-1;
    up=dw;
);
r:=dw+1;
as={r-2}|as;
as=as|{n-(d-1)*(t-1)};
as=as|{n-d*(t-1)+2*t};

k:=d-r+1;
while as#(d-k+1-dw)-as#(d-k-dw)<t+1 do k=k-1;

tmacexp:={};
if k==-1 then tmacexp={{n-d*(t-1),d+1}}
else (
--    for j in r..(d-k) do tmacexp={{as#(j-dw),j}}|tmacexp;
--    tmacexp={{as#(d-k-dw)-2*t+1,d-k+1}}|tmacexp;
--    for j in (d+1-k)..d do tmacexp={{as#(j-dw)-t+1,j+1}}|tmacexp;
    tmacexp=reverse(for j in (d+1-k)..d list {as#(j-dw)-t+1,j+1})|{{as#(d-k-dw)-2*t+1,d-k+1}}|reverse(for j in r..(d-k) list {as#(j-dw),j}); 
);

if not opts.Shift then select(macexp,x->x#0>=0)
else select(tmacexp,x->x#0>=0)
)

------------------------------------------------------------------------
-- whether the given sequence is a ft-vector
------------------------------------------------------------------------
isFTVector = method(TypicalValue=>Boolean)
isFTVector(Ring,List,ZZ) := (S,f,t) -> (
n:=numgens S;
f=join(f,((n+t-1)//t-#f+1):0);

if f#0>1 or f#1>n or #f-2>(n+t-1)//t then return false
else (
    for k from 1 to #f-2 do (
        val:=solveBinomialExpansion tMacaulayExpansion(f#k,n,k,t,Shift=>true);
        if f#(k+1)<0 or f#(k+1)>val then return false
    );
);
true
)

----------------------------------------------------------------------------------
-- return the t-lex ideal with the ft-vector
----------------------------------------------------------------------------------
tLexIdeal = method(TypicalValue=>Ideal)
tLexIdeal (Ring, List, ZZ) := (S,f,t) -> (
n:=numgens S;
ind:=0;
l:={};
ltempold:={};
if isFTVector(S,f,t) then (
    f=join(f,((n+t-1)//t-#f+1):0);

    for d from 0 to #f-1 do (
        completeLex:=tVeroneseSet(S,d,t);
        shad:=tShadow(ltempold,t);
        if shad=={} then shad={0_S};
        ind=binomial(n-(t-1)*(d-1),d)-f#d;
        ltemp:=take(completeLex,ind);
        ltemp=rsort toList(set ltemp - set shad);
        l=join(l,ltemp);
        ltempold=unique select(join(ltemp,shad),x->x!=0);
    );
    if #l==0 then l={0_S};
) else error "expected a valid ft-vector";
ideal l
)

 ----------------------------------------------------------------------------------
-- return the t-lex ideal sharing the same ft-vector of I
----------------------------------------------------------------------------------
tLexIdeal (Ideal,ZZ) := (I,t) -> if isTStronglyStableIdeal(I,t) then tLexIdeal(ring I, fTVector(I,t), t) else error "expected a t-strongly stable ideal"


---------------------------------------------------------------------
-- whether a t-spread ideal is t-spread lex
---------------------------------------------------------------------
isTLexIdeal = method(TypicalValue=>Boolean)
isTLexIdeal (Ideal,ZZ) := (I,t) -> (
S:=ring I;
SL:=newRing(S, MonomialOrder=>Lex);
RM:=map(SL,S);
I=RM ideal mingens I;
degs:=sort flatten degrees I;

if isTSpread(I,t) then        -- or I==ideal(0_SL)
    for d from degs#0 to last degs do (
        monI:=select(rsort flatten entries compress (super basis(d,I)),x->isTSpread(x,t));
        monLex:=take(tVeroneseSet(SL,d,t),#monI);
        if monI!=monLex then return false;
    )
else error "expected a t-spread ideal";
true
)

-------------------------------------------------------------------------------------------
--  Compute the Borel-segment of monomials between ini and fin
-------------------------------------------------------------------------------------------
tStronglyStableSeg = method(TypicalValue=>List)
tStronglyStableSeg (RingElement,RingElement,ZZ) := (ini,fin,t) -> (
S:=ring fin;
n:=numgens S;
l:={};
sup:=index\support fin;
inf:=index\support ini;

if isTSpread(ini,t) and isTSpread(fin,t) and (degree ini)#0 == (degree fin)#0 and ring ini===S then (
    if min(sup-inf)>=0 then (
        l={product(inf/(x->S_x))};
        while inf!=sup do(
            j:=#sup-1;
            while j>=0 and (inf#j)+1>sup#j do (
                j=j-1;
            );
            g:=(inf#j)+1;
            inf=take(inf,j);
            k:=#sup-j;
            while k>0 do (
                inf=inf|{g};
                k=k-1;
                g=g+t;
            );
            l=join(l,{product(inf/(x->S_x))});
        );
    ) else error "expected the first monomial greater than the second one with respect to Borel order";
) else error "expected t-spread monomials of the same degree";
l
)

---------------------------------------------------------------------
-- whether a t-spread set is a t-spread strongly stable segment
---------------------------------------------------------------------
isTStronglyStableSeg = method(TypicalValue=>Boolean)
isTStronglyStableSeg (List,ZZ) := (l,t) -> (
S:=ring l#0;
SL:=newRing(S, MonomialOrder=>Lex);
RM:=map(SL,S);
l=RM\l;
deg:=sort flatten degree l#0;

if unique(l/(x->isTSpread(x,t)))=={true} then (
    if tStronglyStableSeg(max l, min l,t)!= rsort l then return false;
) else error "expected t-spread monomials";
true
)

-------------------------------------------------------------------------------------------
--  Compute the Borel monomials generated by mon
-------------------------------------------------------------------------------------------
tStronglyStableMon = method(TypicalValue=>List)
tStronglyStableMon (RingElement,ZZ) := (mon,t) -> (
S:=ring mon;
l:={};
if isTSpread(mon,t) then (
    ini:=product for i to ((degree mon)#0-1) list S_(i*t);
    l=tStronglyStableSeg(ini,mon,t);
) else error "expected a t-spread monomial";
l
)

-------------------------------------------------------------------------------------------
--  Count the number of t-strongly stable monomials generated by mon
-------------------------------------------------------------------------------------------
countTStronglyStableMon = method(TypicalValue=>ZZ)
countTStronglyStableMon (RingElement,ZZ) := (mon,t) -> (
if isTSpread(mon,t) then (
    supp:=index \ support mon;
    m:=max supp;
    d:=#supp;
    p:=d-2;
    c:=0;
    ind:=new MutableList from toList(d-1:1);

    while ind#0<=supp#0+1 do (
        c=c+m+1-(d-1)*(t-1)- sum toList ind;
        ind#p=ind#p+1;
        while p>0 and ind#p > (supp#p+1)-(sum take(toList ind,p)+p*t-p+1)+1 do (
            ind#p=1;
            p=p-1;
            ind#p=ind#p+1;
        );
        p=d-2;
    );
) else error "expected a t-spread monomial";
c
)

-------------------------------------------------------------------------------------------
--  Compute the Borel ideal generated by I
-------------------------------------------------------------------------------------------
tStronglyStableIdeal = method(TypicalValue=>Ideal)
tStronglyStableIdeal (Ideal,ZZ) := (I,t) -> (
S:=ring I;
n:=numgens S;
SL:=newRing(S, MonomialOrder=>{Weights=>toList(1..n)});
RM:=map(SL,S);
IRM:=map(S,SL);
if isTSpread(I,t) then (
    gen:=flatten entries mingens RM I;
    l:=unique flatten for x in gen list tStronglyStableMon(x,t);
) else error "expected a t-spread ideal";
IRM trim ideal l
)

-------------------------------------------------------------------------------------------
--  whether a t-spread ideal is Borel
-------------------------------------------------------------------------------------------
isTStronglyStableIdeal = method(TypicalValue=>Boolean)
isTStronglyStableIdeal (Ideal,ZZ) := (I,t) -> (
S:=ring I;
n:=numgens S;

if isTSpread(I,t) then (
    gen:=flatten entries mingens I;
    l:={};
    for x in gen do (
        l=unique join(l,tStronglyStableMon(x,t));
        if #select(l/(x->x%I),x->x!=0)>0 then return false;
    );
) else error "expected a t-spread ideal";
true
)


-------------------------------------------------------------------------------------------
-- returns the corners of the extremal Betti numbers of I
----------------------------------------------------------------------------------------------
tExtremalBettiCorners = method(TypicalValue=>List)
tExtremalBettiCorners (Ideal,ZZ) := (I,t) -> (
gr:=unique flatten degrees I;
dseq:={};
corners:={};

if isTStronglyStableIdeal(I,t) then (
    for d in gr do (
        l:=select(flatten entries gens I,x->(degree x)#0==d);
        m:=max apply( l,x->support x / index //max+1);
        dseq=join(dseq,{(m-t*(d-1)-1,d)});
    );
    while #dseq>1 do (
        if dseq#0#0 > max (take(dseq,{1,#dseq}) / (y->y#0)) then
            corners=join(corners,{dseq#0});
        dseq=take(dseq,{1,#dseq});
    );
) else error "expected a t-strongly stable ideal";
join(corners,{dseq#0})
)

-------------------------------------------------------------------------------------------
-- returns the basic t-spread monomials with extremal Betti numbers assigned as positions as values
----------------------------------------------------------------------------------------------
tExtremalBettiMonomials = method(TypicalValue=>List)
tExtremalBettiMonomials (Ring,List,List,ZZ) := (S,corners,a,t) -> (
n:=numgens S;
SL:=newRing(S, MonomialOrder=>Lex);
vrs:=flatten entries vars SL;
RM:=map(SL,S);
IRM:=map(S,SL);
r:=#corners;
cond:=r<n and #corners==r and #a==r;
cond=cond and rsort unique (corners / (x->x#0))== corners / (x->x#0);
cond=cond and sort unique (corners / (x->x#1))== corners / (x->x#1);
cond= cond and max(corners / (x->x#0+x#1))<=n;

if cond then (
    mu:=corners#0#0+t*(corners#0#1-1)+1;
    g:=mu-1;
    bound:=binomial(corners#0#0+corners#0#1-1,corners#0#1-1);
    if a#0>bound then error "Ideal does not exists";
    mon:=product((t*toList(0..corners#0#1-2)/(i->SL_i))|{SL_g});
    l:={mon};
    for j from 2 to a#0 do (
        try mon=tNextMon(mon,t,FixedMax=>true) then
            l=join(l,{mon})
        else error "no t-strongly stable ideal exists";
    );
    for i from 1 to r-1 do (
        mu=corners#i#0+t*(corners#i#1-1)+1;
        g=mu-1;
        --last monomial of the shadow
        try mon=tLastMon(mon,corners#i#1-corners#(i-1)#1,t,MaxInd=>g+1)
        else error "no t-strongly stable ideal exists";
        --first monomial over the shadow
        for j from 1 to a#i do (
            try mon=tNextMon(mon,t,FixedMax=>true) then
                l=join(l,{mon})
            else error "no t-strongly stable ideal exists";
        );
    );
) else error "the hypotheses do not hold";
IRM\l
)

------------------------------------------------------------------------
-- Compute the sum of a binomial expansion.
------------------------------------------------------------------------
solveBinomialExpansion = method(TypicalValue=>ZZ)
solveBinomialExpansion List := l -> sum apply(l,x->binomial(toSequence x))

------------------------------------------------------------------------
-- returns the initial degree of a graded ideal
------------------------------------------------------------------------
initialDegree = method(TypicalValue=>ZZ)
initialDegree Ideal := I -> (
if isHomogeneous I then return min flatten degrees ideal mingens I
else error "expected a graded ideal";
)

-------------------------------------------------------------------------------------------
-- returns the initial ideal of an ideal
----------------------------------------------------------------------------------------------
initialIdeal = method(TypicalValue=>Ideal)
initialIdeal Ideal := I -> ideal rsort leadTerm I

-------------------------------------------------------------------------------------------
-- returns the minimal Betti numbers of the quotient S/I
----------------------------------------------------------------------------------------------
minimalBettiNumbers = method(TypicalValue=>BettiTally)
minimalBettiNumbers Ideal := I -> betti res ideal flatten entries mingens I

-------------------------------------------------------------------------------------------
-- Computes the minimal Betti numbers of I
----------------------------------------------------------------------------------------------
minimalBettiNumbersIdeal = method(TypicalValue=>BettiTally)
minimalBettiNumbersIdeal Ideal := I -> betti res pushForward(map(ring I,ring I), image mingens I)

beginDocumentation()
-------------------------------------------------------
--DOCUMENTATION ExteriorIdeals
-------------------------------------------------------

document {
Key => {TSpreadIdeals},
Headline => "A package for working with t-spread ideals of polynomial rings",
"TSpreadIdeals is a package for creating and manipulating t-spread ideals of polynomial rings. Let ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///, " a monomial of ", TEX///$S=K[x_1,\ldots,x_n]$///, ", with ", TEX///$1\le i_1\le i_2\le\dots\le i_d\le n$///, ". The monomial ", TT "u", " is called ", TEX///$t$///, "-spread if ", TEX///$i_{j+1}-i_j\ge t$///, " for all ", TEX///$j\in [d-1].$///, BR{},
"A list of monomials is ", TT "t", "-spread if all the monomials of the list are ", TT "t", "-spread. A monomial ideal is called ", TT "t", "-spread if it is generated by ", TT "t", "-spread monomals.",BR{},
"Some references can be found in the papers ", HREF("https://arxiv.org/abs/2110.11801","Computational methods for t-spread monomial ideals"), " and ", HREF("https://arxiv.org/abs/2105.11944","A numerical characterization of the extremal Betti numbers of t-spread strongly stable ideals"), ".",
Subnodes => {
"T-spread managing",
    TO "tLastMon",
    TO "tPascalIdeal",
    TO "tSpreadList",
    TO "tSpreadIdeal",
    TO "isTSpread",
    TO "tShadow",
"T-lex structures",
    TO "tNextMon",
    TO "tLexSeg",
    TO "isTLexSeg",
    TO "tLexMon",
    TO "countTLexMon",
    TO "tVeroneseSet",
    TO "tVeroneseIdeal",
    TO "isTLexIdeal",
"T-strongly stable structures",
    TO "tStronglyStableSeg",
    TO "isTStronglyStableSeg",
    TO "tStronglyStableMon",
    TO "countTStronglyStableMon",
    TO "tStronglyStableIdeal",
    TO "isTStronglyStableIdeal",
"Kruskal-Katona's Theorem",
    TO "tMacaulayExpansion",
    TO "fTVector",
    TO "isFTVector",
    TO "tLexIdeal",
"Extremal Betti numbers",
    TO "tExtremalBettiCorners",
    TO "tExtremalBettiMonomials",
"Services",
    TO "solveBinomialExpansion",
    TO "initialDegree",
    TO "initialIdeal",
    TO "minimalBettiNumbers",
    TO "minimalBettiNumbersIdeal",
"Options",
    TO "Shift",
    TO "FixedMax",
    TO "MaxInd"
},
}

document {
Key => {tNextMon,(tNextMon,RingElement,ZZ)},
Headline => "give the t-lex successor of a given t-spread monomial",
Usage => "tNextMon(u,t)",
Inputs => {"u" => {"a t-spread monomial of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {RingElement => {"the greatest ", TT "t", "-spread monomial less than ", TT "u", ", with respect to lexcografic order"}},
"the function ", TT "tNextMon(u,t)", " gives the ", TT "t" , "-lex successor of ", TT "t", ", that is, the greatest ", TT "t" , "-spread monomial less than ", , TT "u", " with respect to ", TEX///$>_\mathrm{slex}.$///,BR{},
"If ", TT "FixedMax", " is ",TT "true", " the function ", TT "tNextMon(u,t,FixedMax=>true)", " gives the ", TT "t" , "-lex successor for which the maximum of the support is ", TEX///$\max\textrm{supp}(\texttt{u}).$///,BR{},
"Given a ", TEX///$t$///, "-spread monomial ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///,", we define ", TEX///$\textrm{supp}(u)=\{i_1,i_2,\ldots, i_d\}$///, ".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_16]
    tNextMon(x_2*x_6*x_10*x_13,3)
    tNextMon(x_2*x_6*x_10*x_13,3,FixedMax=>true)
///,
SeeAlso =>{FixedMax, tLexSeg, tLexIdeal},
}

document {
Key => {tLastMon,(tLastMon,RingElement,ZZ,ZZ)},
Headline => "give the last t-spread monomial of the Borel shadow a given t-spread monomial",
Usage => "tLastMon(u,gap,t)",
Inputs => {"u" => {"a t-spread monomial of a polynomial ring"},
           "gap" => {"a positive integer representing the difference between the degrees"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {RingElement => {"the smallest ", TT "t", "-spread monomial of the Borel shadow of ", TT "u", ", with respect to lexcografic order"}},
"the function ", TT "tLastMon(u,gap,t)", " gives the smallest ", TT "t" , "-spread monomials of the Borel shadow iterated ", TT "gap", "times of ", TT "u", ", that is, the smallest monomial of ", TEX///$B_\texttt{t}\{\texttt{u}\}$///, ", with respect to ", TEX///$>_\mathrm{slex}.$///,BR{},
"If ", TT "MaxInd", " is greater than -1 the function ", TT "tLastMon(u,gap,t,MaxInd=>m)", " gives the smallest ", TT "t" , "-spread monomial for which the maximum of the support is ", TT "m", BR{},
"The Borel shadow of a ", TT "t", "-spread monomial ", TT "u", ", is defined as the shadow of the strongly stable set generated by ", TT "u", ". To work in a ", TT "t", "-spread contest, the Borel shadow of ", TT "u", " is the ", TT "t", "-shadow of ", TEX///$B_\texttt{t}\{u\}$///, ".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_16]
    tLastMon(x_2*x_6*x_10*x_13,1,3)
    tLastMon(x_2*x_6*x_10*x_13,1,3,MaxInd=>14)
///,
SeeAlso =>{FixedMax, tShadow},
}

document {
Key => {tLexSeg,(tLexSeg,RingElement,RingElement,ZZ)},
Headline => "give the t-lex segment with given extremes",
Usage => "tLexSeg(v,u,t)",
Inputs => {"v" => {"a t-spread monomial of a polynomial ring"},
           "u" => {"a t-spread monomial of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the set of all the ", TT "t", "-spread monomials smaller than ", TT "v", " and greater than ", TT "u", ", with respect to lexcografic order"}},
"the function ", TT "tLexSeg(v,u,t)", " gives the ", TT "t" , "-lex segment defined by ", TT "v", " and ", TT "u", ", that is, the set of all the ", TT "t" , "-spread monomials smaller than ", , TT "v", " and greater than ", TT "u", ", with respect to ", TEX///$>_\mathrm{slex}.$///,BR{},
"This function iteratively uses the method ", TT "tNextMon(u,t)", " until the returned monomial is ",TT "u", ".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_14]
    tLexSeg(x_2*x_5*x_10*x_13,x_2*x_6*x_9*x_12,2)
    tLexSeg(x_2*x_5*x_10*x_13,x_2*x_6*x_9*x_12,3)
///,
SeeAlso =>{tNextMon, tLexIdeal},
}

document {
Key => {isTLexSeg,(isTLexSeg,List,ZZ)},
Headline => "whether a set of t-spread monomials is a t-lex segment",
Usage => "isTLexSeg(l,t)",
Inputs => {"l" => {"a list of t-spread monomials of a polynomial ring"}
},
Outputs => {Boolean => {"whether the list ", TT "l", " of monomials is a t-lex segment"}},
"the function ", TT "isTLexSeg(l,t)", " is ", TT "true", " whether the list of monomials ", TT "l" , " is a ", TT "t", "-lex segment, that is, the set of all the ", TT "t" , "-spread monomials smaller than the maximum of ", TT "l", " and greater than the minimum of ", TT "l", ", with respect to ", TEX///$>_\mathrm{slex}.$///,
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_7]
    isTLexSeg({x_1*x_4*x_7,x_1*x_5*x_7,x_2*x_4*x_6,x_2*x_4*x_7,x_2*x_5*x_7},2)
    isTLexSeg({x_1*x_4*x_7,x_2*x_4*x_6,x_2*x_4*x_7,x_2*x_5*x_7},2)
///,
SeeAlso =>{tLexSeg, tNextMon},
}

document {
Key => {tLexMon,(tLexMon,RingElement,ZZ)},
Headline => "give the smallest initial t-lex segment containing a given monomial",
Usage => "tLexMon(u,t)",
Inputs => {"u" => {"a t-spread monomial of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the set of all the ", TT "t", "-spread monomials greater than ", TT "u", ", with respect to lexcografic order"}},
"the function ", TT "tLexMon(u,t)", " gives the initial ", TT "t" , "-lex segment defined by ", TT "u", ", that is, the set of all the ", TT "t" , "-spread monomials greater than ", TT "u", ", with respect to ", TEX///$>_\mathrm{slex}.$///,BR{},
"This function calls the method ", TT "tLexSeg(v,u,t)", ", where ",TT "v", " is the greatest ", TT "t" , "-spread monomial of the polynomial ring.", " Let ", TEX///$S=K[x_1,\ldots,x_n]$///, " and ", TEX///$u\in M_{n,d,t}$///, ", then ", TEX///$v=x_1x_{1+t}\cdots x_{1+(d-1)t}$///, ".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_9]
    tLexMon(x_2*x_5*x_8,2)
    tLexMon(x_2*x_5*x_8,3)
///,
SeeAlso =>{tNextMon, tLexSeg, isTLexSeg},
}

document {
Key => {countTLexMon,(countTLexMon,RingElement,ZZ)},
Headline => "give the cardinality of the smallest initial t-lex segment containing a given monomial",
Usage => "countTLexMon(u,t)",
Inputs => {"u" => {"a t-spread monomial of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {ZZ => {"the number of all the ", TT "t", "-spread monomials greater than ", TT "u", ", with respect to lexcografic order"}},
"the function ", TT "countTLexMon(u,t)", " gives the cardinality of ", TEX///$L_t\{u\}$///, ", the initial ", TT "t" , "-lex segment defined by ", TT "u", ", that is, the number of all the ", TT "t" , "-spread monomials greater than ", TT "u", ", with respect to ", TEX///$>_\mathrm{slex}.$///,BR{},
"If ", TT "FixedMax", " is ",TT "true", " the function ", TT "countTLexMon(u,t,FixedMax=>true)", " gives the number of all the ", TT "t" , "-spread monomials having maximum of the support equal to ", TEX///$\max\textrm{supp}(u)$///, " and greater than ", TT "u", ", with respect to ", TEX///$>_\mathrm{slex}.$///,BR{},
"Given a ", TEX///$t$///, "-spread monomial ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///,", we define ", TEX///$\textrm{supp}(u)=\{i_1,i_2,\ldots, i_d\}$///, ".", BR{},
"This method is not constructive. It uses a theoretical result to obtain the cardinality as the sum of suitable binomial coefficients. The procedure only concerns ", TEX///$\textrm{supp}(\texttt{u}),$///, " that is, the set ", TEX///$\{i_1,i_2,\ldots, i_d\}$///, ", when ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///, " is a ", TEX///$t$///, "-spread monomial.",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_9]
    countTLexMon(x_2*x_5*x_8,2)
    countTLexMon(x_2*x_5*x_8,3)
///,
SeeAlso =>{tLexMon},
}

document {
Key => {tVeroneseSet,(tVeroneseSet,Ring,ZZ,ZZ)},
Headline => "give the Veronese set of t-spread monomials of a given degree",
Usage => "tVeroneseSet(S,d,t)",
Inputs => {"S" => {"a polynomial ring"},
           "d" => {"a nonnegative integer that identifies the degree of the monomials"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the set of all the ", TT "t", "-spread monomials of ", TT "S", " of degree ", TT "d"}},
"the function ", TT "tVeroneseSet(S,d,t)", " gives the Veronese set of ", TT "t" , "-spread monomials of degree ", TT "d", ", that is, the set of all the ", TT "t" , "-spread monomials of ", TT "S", " of degree ", TT "d",
"This function calls the method ", TT "tLexSeg(v,u,t)", ", where ",TT "v", " is the greatest ", TT "t" , "-spread monomial of the polynomial ring and ", TT "u", " the smallest one. If ", TEX///$S=K[x_1,\ldots,x_n]$///, " then ", TEX///$v=x_1x_{1+t}\cdots x_{1+(d-1)t}$///, " and ", TEX///$u=x_{n-(d-1)t}\cdots x_{n-t} x_{n}$///, ". This set is also denoted by ", TEX///$M_{n,d,t}$///, ".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_8]
    tVeroneseSet(S,3,2)
    tVeroneseSet(S,3,3)
///,
SeeAlso =>{tNextMon, tLexSeg, isTLexSeg, tLexMon, tVeroneseIdeal},
}

document {
Key => {tVeroneseIdeal,(tVeroneseIdeal,Ring,ZZ,ZZ)},
Headline => "give the Veronese ideal of t-spread monomials of a given degree",
Usage => "tVeroneseIdeal(S,d,t)",
Inputs => {"S" => {"a polynomial ring"},
           "d" => {"a nonnegative integer that identifies the degree of the monomials"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {Ideal => {"the ideal generated by all the ", TT "t", "-spread monomials of ", TT "S", " of degree ", TT "d"}},
"the function ", TT "tVeroneseIdeal(S,d,t)", " returns the Veronese ideal of ", TT "t" , "-spread monomials of degree ", TT "d", ", that is, the ideal generated by all the ", TT "t" , "-spread monomials of ", TT "S", " of degree ", TT "d",
"This function calls the method ", TT "tVeroneseSet(S,d,t)", ".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_8]
    tVeroneseIdeal(S,3,2)
    tVeroneseIdeal(S,3,4)
///,
SeeAlso =>{tNextMon, tLexSeg, isTLexSeg, tLexMon, tVeroneseSet},
}

document {
Key => {tPascalIdeal,(tPascalIdeal,Ring,ZZ)},
Headline => "give the Pascal ideal of t-spread monomials of a given polynomial ring",
Usage => "tPascalIdeal(S,t)",
Inputs => {"S" => {"a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {Ideal => {"the Pascal ideal of ", TT "t", "-spread monomials of ", TT "S"}},
"the function ", TT "tPascalIdeal(S,t)", " returns the Pascal ideal of ", TT "t" , "-spread monomials of ", TT "S", ", that is, the ideal generated by the minimun number of ", TT "t" , "-spread monomials of ", TT "S", " with the greatest possible degrees and such that the supports of the generators are pairwise disjoint.",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_10]
    tPascalIdeal(S,3)
    tPascalIdeal(S,4)
///,
}

document {
Key => {tSpreadList,(tSpreadList,List,ZZ)},
Headline => "give the set of all t-spread monomials of a given list",
Usage => "tSpreadList(l,t)",
Inputs => {"l" => {"a list of t-spread monomials of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the set of all the ", TT "t", "-spread monomials belonging to ", TT "l"}},
"the function ", TT "tSpreadList(l,t)", " gives the list of all the ", TT "t" , "-spread monomials ", TT "l", ".",BR{},
"This function calls the method ", TT "isTSpread(u,t)", " to check whether the monomial ", TT "u", " is ", TT "t" , "-spread, for all the monomials that belong to the list ", TT "l", ".", BR{},
"Let ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///, " a monomial of ", TEX///$S=K[x_1,\ldots,x_n]$///, ", with ", TEX///$1\le i_1\le i_2\le\dots\le i_d\le n$///, ". The monomial ", TT "u", " is called ", TEX///$t$///, "-spread if ", TEX///$i_{j+1}-i_j\ge t$///, " for all ", TEX///$j\in [d-1]$///, ".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_14]
    l={x_3*x_7*x_10*x_14, x_1*x_5*x_9*x_13}
    tSpreadList(l,4)
    tSpreadList(l,5)
///,
SeeAlso =>{tSpreadIdeal, isTSpread},
}

document {
Key => {tSpreadIdeal,(tSpreadIdeal,Ideal,ZZ)},
Headline => "give the ideal generated by the t-spread monomials which are among the generators of a given ideal",
Usage => "tSpreadIdeal(I,t)",
Inputs => {"I" => {"a graded ideal of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {Ideal => {"the ideal generated by all the ", TT "t", "-spread monomials belonging to the generator set of the ideal ", TT "I"}},
"the function ", TT "tSpreadIdeal(I,t)", " gives the ideal generated by all the ", TT "t" , "-spread monomials which are among the generators of the ideal ", TT "I", ".",BR{},
"This function calls the method ", TT "tSpreadList(l,t)", " to sieve the ", TT "t" , "-spread monomials from the list ", TT "l", ", of the generators of the ideal ", TT "I", ".", BR{},
"Let ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///, " a monomial of ", TEX///$S=K[x_1,\ldots,x_n]$///, ", with ", TEX///$1\le i_1\le i_2\le\dots\le i_d\le n$///, ". The monomial ", TT "u", " is called ", TEX///$t$///, "-spread if ", TEX///$i_{j+1}-i_j\ge t$///, " for all ", TEX///$j\in [d-1]$///, ". A monomial ideal is called ", TT "t", "-spread if it is generated by ", TT "t", "-spread monomals.",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_14]
    I=ideal {x_3*x_7*x_10*x_14, x_1*x_5*x_9*x_13}
    tSpreadIdeal(I,3)
    tSpreadIdeal(I,4)
///,
SeeAlso =>{tSpreadList, isTSpread},
}

document {
Key => {isTSpread,(isTSpread,RingElement,ZZ),(isTSpread,List,ZZ),(isTSpread,Ideal,ZZ)},
Headline => "whether a monomial, a list of monomials or a monomial ideal is t-spread",
Usage => "isTSpread(u,t), isTSpread(l,t) or isTSpread(I,t)",
Inputs => {"u" => {"a monomial of a polynomial ring"},
           "l" => {"a list of monomials of a polynomial ring"},
           "I" => {"a monomial ideal of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {Boolean => {"whether a given monomial ", TT "u", ", a list of monomials " , TT "l", " or a monomial ideal ", TT "I", " is ", TT "t", "-spread"}},
"the function ", TT "isTSpread(-,t)", " has three overloading changing for the first parameter. It checks whether a given monomial ", TT "u", ", a list of monomials ", TT "l", " or a monomial ideal ", TT "I", " is ", TT "t" , "-spread.",BR{},
"Let ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///, " a monomial of ", TEX///$S=K[x_1,\ldots,x_n]$///, ", with ", TEX///$1\le i_1\le i_2\le\dots\le i_d\le n$///, ". The monomial ", TT "u", " is called ", TEX///$t$///, "-spread if ", TEX///$i_{j+1}-i_j\ge t$///, " for all ", TEX///$j\in [d-1].$///, BR{},
"A list of monomials is ", TT "t", "-spread if all the monomials of the list are ", TT "t", "-spread. A monomial ideal is called ", TT "t", "-spread if it is generated by ", TT "t", "-spread monomals.",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_14]
    l={x_3*x_7*x_10*x_14, x_1*x_5*x_9*x_13}
    isTSpread(l#0,3)
    isTSpread(l,3)
    isTSpread(ideal l,3)
    isTSpread(ideal l,4)
///
}

document {
Key => {tShadow,(tShadow,RingElement,ZZ),(tShadow,List,ZZ)},
Headline => "give the t-spread shadow of a given t-spread monomial or a given list of t-spread monomials",
Usage => "tShadow(u,t)",
Inputs => {"u" => {"a t-spread monomial of a polynomial ring"},
           "l" => {"a list of t-spread monomials of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the list of all the ", TT "t", "-spread monomial of the shadow of the ", TT "t", "-spread monomial ", TT "u", " or of the list ", TT "l"}},
"the function ", TT "tShadow(u,t)", " gives the ", TT "t" , "-spread shadow of ", TT "u", ", that is, the set of all the ", TT "t" , "-spread monomials of the shadow of ", TT "u", ". The overloading function ", TT "tShadow(l,t)", " gives the ", TT "t" , "-spread shadow of ", TT "l", ", that is, the set of all the ", TT "t" , "-spread monomials of the shadow of each ", TT "t", "-spread monomial belonging to ", TT "l", ".",BR{},
"Let ", TEX///$S=K[x_1,\ldots,x_n]$///, " and ", TEX///$u\in M_{n,d,t}$///, ", that is, ", TT "u", " is a ", TEX///$t$///, "-spread monomial of degree ", TEX///$d$///, ". The ", TT "t", "-spread shadow of ", TT "u", ", is defined as ", TEX///$\mathrm{Shad}_t(u)=\{ux_i\ :\ i\in [n]\}\cap M_{n,d+1,t}$///, ". The algorithm is optimized for the ", TEX///$t$///, "-spread environment.",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_14]
    u=x_2*x_6*x_10
    tShadow(u,3)
    tShadow(u,4)
    l={x_3*x_6*x_10, x_1*x_5*x_9}
    tShadow(l,3)
    tShadow(l,4)
///,
SeeAlso =>{isTSpread},
}

document {
Key => {tMacaulayExpansion,(tMacaulayExpansion,ZZ,ZZ,ZZ,ZZ)},
Headline => "compute the t-Macaulay expansion of a positive integer",
Usage => "tMacaulayExpansion(a,n,d,t)",
Inputs => {"a" => {"a positive integer to be expanded"},
           "n" => {"a positive integer that identifies the number of indeterminates"},
           "d" => {"a positive integer that identifies the degree"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"pairs of positive integers representing the ", TT "d", "-th (shifted) t-Macaulay expansion of ", TT "a", " with a given ", TT "n"}},
"Given four positive integers ", TT "(a,n,d,t)", " there is a unique expression of ", TT "a", " as a sum of binomials ", TEX///$a=\binom{a_d}{d} + \binom{a_{d-1}}{d-1} + \cdots + \binom{a_j}{j}.$///, " where ", TEX///$a_i > a_{i-1} > \cdots > a_j > j >= 1.$///,BR{},
"If the optional parameter ", TT "Shift", " is ", TT "true", ", then the method ", TT "tMacaulayExpansion(a,n,d,t,Shift=>true)", " returns the shifted t-Macaulay expansion of ", TT "a", ", that is, ", TEX///$a^{(d)}=\binom{a_d}{d+1} + \binom{a_{d-1}}{d} + \cdots + \binom{a_j}{j+1}.$///,
"To obtain the sum of the binomial coefficients represented in the output list, one can use the method ", TT "solveBinomialExpansion.",
PARA {"Examples:"},
EXAMPLE lines ///
    tMacaulayExpansion(50,12,2,1)
    tMacaulayExpansion(50,12,2,1,Shift=>true)
    tMacaulayExpansion(50,12,2,2,Shift=>true)
///,
SeeAlso =>{solveBinomialExpansion, isFTVector, Shift}
}

document {
Key => {fTVector,(fTVector,Ideal,ZZ)},
Headline => "compute the ft-vector of a given t-spread ideal of a polynomial ring",
Usage => "fTVector(I,t)",
Inputs => {"I" => {"a t-spread ideal of polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"a list of nonnegative integers representing the ft-vector of the t-spread ideal ", TT "I"}},
"Let ", TT "I", " be a ", TT "t", "-spread ideal of the polynomial ring ", TEX///$S=K[x_1,\ldots,x_n]$///, " One can define the ", TEX///$f_\texttt{t}$///, "-vector of ", TT "I", " as ", TEX///$f_\texttt{t}(\texttt{I})=\left( f_{\texttt{t},-1}(\texttt{I}), f_{\texttt{t},0}(\texttt{I}), \ldots, f_{\texttt{t},j}(\texttt{I}), \ldots \right),$///, BR{},
"where ", TEX///$f_{\texttt{t},j-1}(\texttt{I})=|[S_j]_t|-|[I_j]_t|$///, " and ", TEX///$[I_j]_t$///, " is the ", TT "t", "-spread part of the ", TEX///$j$///, "-th graded component of ", TT "I", ".",BR{},
"In an equivalent way, let ", TEX///$\Delta$///, " be the associated simplicial complex of ", TT "I", ". So, ", TEX///$f_{\texttt{t},j}(\texttt{I})$///, " is the cardinality of the set ", TEX///$\left\{F\ :\ F \text{ is a } j \text{-dimensional face in } \Delta \text{ and } x_F=\prod_{i\in F}{x_i} \text{ is a } t \text{-spread monomial}\right\}.$///,
PARA {"Example:"},
EXAMPLE lines ///
    S=QQ[x_1..x_8]
    fTVector(ideal {x_1*x_3,x_2*x_5*x_7},1)
    fTVector(ideal {x_1*x_3,x_2*x_5*x_7},2)
///,
SeeAlso =>{isFTVector}
}

document {
Key => {isFTVector,(isFTVector,Ring,List,ZZ)},
Headline => "wheter a given list of nonnegative intengers is the ft-vector of a t-strongly stable ideal of a given polynomial ring",
Usage => "isFTVector(S,f,t)",
Inputs => {"S" => {"a polynomial ring"},
           "f" => {"a list of nonnegative integers"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {Boolean => {"wheter the list ", TT "f", " represents the ft-vector of a t-strongly stable ideal of ", TT "S"}},
"Let ", TEX///$\texttt{S}=K[x_1,\ldots,x_n]$///, ", ", TEX///$\texttt{t}\geq 1$///, " and let ", TEX///$\texttt{f}=(f(0),f(1),\ldots,f(d),\ldots)$///, " be a sequence of nonnegative integers. The sequence ", TT "f", " is the ", TEX///$\texttt{f}_\texttt{t}$///, "-vector of a ", TT "t", "-spread ideal of ", TT "S", " if the following conditions hold:",BR{},
TEX///$f(0)\leq 1,\ f(1)\leq n \text{ and } f(d+1)\leq \texttt{tMacaulayExpansion(f(d),n,d,t,Shift=>true)}$///, " for all ", TEX///$d>1.$///, BR{},
"Let ", TT "I", " be a ", TT "t", "-spread ideal of the polynomial ring ", TEX///$S=K[x_1,\ldots,x_n]$///, " One can define the ", TEX///$f_\texttt{t}$///, "-vector of ", TT "I", " as ", TEX///$f_\texttt{t}(\texttt{I})=\left( f_{\texttt{t},-1}(\texttt{I}), f_{\texttt{t},0}(\texttt{I}), \ldots, f_{\texttt{t},j}(\texttt{I}), \ldots \right),$///, BR{},
"where ", TEX///$f_{\texttt{t},j-1}(\texttt{I})=|[S_j]_t|-|[I_j]_t|$///, " and ", TEX///$[I_j]_t$///, " is the ", TT "t", "-spread part of the ", TEX///$j$///, "-th graded component of ", TT "I", ".",
PARA {"Example:"},
EXAMPLE lines ///
    S=QQ[x_1..x_8]
    f={1,8,20,10,0}
    isFTVector(S,f,2)
    S=QQ[x_1..x_7]
    isFTVector(S,f,2)
///,
SeeAlso =>{fTVector, tMacaulayExpansion, solveBinomialExpansion}
}

document {
Key => {tLexIdeal,(tLexIdeal,Ring,List,ZZ),(tLexIdeal,Ideal,ZZ)},
Headline => "returns the t-spread lex ideal with a given ft-vector or with the same ft-vector of a given t-strongly stable ideal",
Usage => "tLexIdeal(S,f,t) or tLexIdeal(I,t)",
Inputs => {"S" => {"a polynomial ring"},
           "f" => {"a sequence of nonnegative integers"},
           "I" => {"a t-strongly stable ideal of a polynomial ring"},
          "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {Ideal => {"the t-spread lex ideal with ft-vector ", TT "f", " or the t-spread lex ideal with the same ft-vector of ", TT "I"}},
"It has been proved that if ", TT "I", " is a ", TT "t", "-strongly stable ideal then a unique ", TT "t", "-lex ideal with the same ", TEX///$f_\texttt{t}$///, "-vector of ", TT "I", " exists.", BR{},
"Let ", TEX///$\texttt{S}=K[x_1,\ldots,x_n]$///, ", ", TEX///$\texttt{t}\geq 1$///,". The method ", TT"tLexIdeal(S,f,t)", " gives the ", TT "t", "-lex ideal of ", TT "S", " with ", TT "f", " as ", TEX///$f_\texttt{t}$///, "-vector, if exists. The overloading method ", TT"tLexIdeal(I,t)", " gives the ", TT "t", "-lex ideal with the same ", TEX///$f_\texttt{t}$///, "-vector of the ", TT "t", "-strongly stable ideal ", TT "I", ".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_8]
    f={1,8,2,0,0}
    I=tLexIdeal(S,f,2)
    fTVector(I,2)==f
    isTLexIdeal(I,2)
    J=tStronglyStableIdeal(ideal {x_1*x_4*x_6},2)
    K=tLexIdeal(J,2)
    fTVector(J,2)==fTVector(K,2)
///,
SeeAlso =>{isTLexIdeal, fTVector, isFTVector, isTStronglyStableIdeal}
}

document {
Key => {isTLexIdeal,(isTLexIdeal,Ideal,ZZ)},
Headline => "whether a given t-spread ideal is t-spread lex",
Usage => "isTLexIdeal(I,t)",
Inputs => {"I" => {"a t-spread ideal of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {Boolean => {"whether the ideal ", TT "I", " is t-spread lex"}},
"Let ", TEX///$\texttt{S}=K[x_1,\ldots,x_n]$///, ", ", TEX///$\texttt{t}\geq 1$///, " and a ", TT "t", "-spread ideal ", TT "I", ". Then ", TT "I", " is called a ", TT "t", "-lex ideal, if ", TEX///$[I_j]_t$///, " is a ", TT "t", "-spread lex set for all ", TEX///$j$///, ".",BR{},
"We recall that ", TEX///$[I_j]_t$///, " is the ", TT "t", "-spread part of the ", TEX///$j$///, "-th graded component of ", TT "I", ". Moreover, a subset ", TEX///$L\subset M_{n,d,t}$///, " is called a ", TT "t", "-lex set if for all ", TEX///$u\in L$///, " and for all ", TEX///$v\in M_{n,d,t}$///, " with ", TEX///$v>_\mathrm{slex} u$///, ", it follows that ", TEX///$u\in L$///,".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_6]
    isTLexIdeal(ideal {x_1*x_3,x_1*x_5},2)
    isTLexIdeal(ideal {x_1*x_3,x_1*x_4,x_1*x_5,x_1*x_6,x_2*x_4,x_2*x_5},2)
///,
SeeAlso =>{tLexIdeal},
}

document {
Key => {tStronglyStableSeg,(tStronglyStableSeg,RingElement,RingElement,ZZ)},
Headline => "give the t-strongly stable segment with the given extremes",
Usage => "tStronglyStableSeg(v,u,t)",
Inputs => {"v" => {"a t-spread monomial of a polynomial ring"},
           "u" => {"a t-spread monomial of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the set of all the ", TT "t", "-spread monomials of the t-stromgly stable set generated by ", TT "u", " and smaller than ", TT "u", ", with respect to lexcografic order"}},
"the function ", TT "tStronglyStableSeg(v,u,t)", " gives the set of ", TT "t" , "-spread monomials belonging to the strongly stable set generated by ", TT "u", " and smaller than ", TT "v", ", that is, ", TEX///$B_t[v,u]=\{w\in B_t\{u\}\ :\ v\geq_\mathrm{slex} w\}.$///,BR{},
"We recall that if ", TEX///$u\in M_{n,d,t}\subset S=K[x_1,\ldots,x_n]$///, " then ", TEX///$B_t\{u\}$///, " is the smallest ",TT "t", "-strongly stable set of monomials of ", TEX///$M_{n,d,t}$///, " containing ", TEX///$u.$///, BR{},
"Moreover, a subset ", TEX///$N\subset M_{n,d,t}$///, " is called a ", TT "t", "-strongly stable set if taking a ", TT "t", "-spread monomial ",TEX///$u\in N$///, ", for all ", TEX///$j\in \mathrm{supp}(u)$///, " and all ", TEX///$i,\ 1\leq i\leq j$///, ", such that ", TEX///$x_i(u/x_j)$///," is a ", TT"t", "-spread monomial, then it follows that ", TEX///$x_i(u/x_j)\in N$///,".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_14]
    tStronglyStableSeg(x_2*x_5*x_9*x_12,x_2*x_6*x_10*x_13,2)
    tStronglyStableSeg(x_2*x_5*x_9*x_12,x_2*x_6*x_10*x_13,3)
///,
SeeAlso =>{isTStronglyStableSeg, tStronglyStableMon},
}

document {
Key => {isTStronglyStableSeg,(isTStronglyStableSeg,List,ZZ)},
Headline => "whether the given list of t-spread monomials is a t-strongly stable segment",
Usage => "isTStronglyStableSeg(l,t)",
Inputs => {"l" => {"a list of t-spread monomials of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {Boolean => {"whether the list ", TT "l", " of ", TT "t", "-spread monomials is a ", TT "t", "-strongly stable segment"}},
"the function ", TT "isTStronglyStableSeg(l,t)", " is ", TT "true", " whether the list of monomials ", TT "l" , " is a ", TT "t", "-strongly stable segment, that is, the set of all the ", TT "t" , "-spread monomials belonging to the strongly stable set generated by the smallest monomial of ", TT "l", " and smaller than the greatest monomial of ", TT "l", ", with respect to ", TEX///$>_\mathrm{slex}.$///,BR{},
"In other words, whether ", TEX///$l=B_t[\max l,\min l]=\{w\in B_t\{\min l\}\ :\ \max l\geq_\mathrm{slex} w\}.$///,BR{},
"We recall that if ", TEX///$u\in M_{n,d,t}\subset S=K[x_1,\ldots,x_n]$///, " then ", TEX///$B_t\{u\}$///, " is the smallest ",TT "t", "-strongly stable set of monomials of ", TEX///$M_{n,d,t}$///, " containing ", TEX///$u.$///, BR{},
"Moreover, a subset ", TEX///$N\subset M_{n,d,t}$///, " is called a ", TT "t", "-strongly stable set if taking a ", TT "t", "-spread monomial ",TEX///$u\in N$///, ", for all ", TEX///$j\in \mathrm{supp}(u)$///, " and all ", TEX///$i,\ 1\leq i\leq j$///, ", such that ", TEX///$x_i(u/x_j)$///," is a ", TT"t", "-spread monomial, then it follows that ", TEX///$x_i(u/x_j)\in N$///,".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_9]
    isTStronglyStableSeg({x_1*x_4*x_7,x_1*x_4*x_8,x_1*x_5*x_8,x_2*x_5*x_8},3)
    isTStronglyStableSeg({x_1*x_4*x_7,x_1*x_4*x_8,x_2*x_5*x_8},3)
///,
SeeAlso =>{tStronglyStableSeg, tStronglyStableMon},
}

document {
Key => {tStronglyStableMon,(tStronglyStableMon,RingElement,ZZ)},
Headline => "give the t-strongly stable set generated by a given monomial",
Usage => "tStronglyStableMon(u,t)",
Inputs => {"u" => {"a t-spread monomial of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the list of all the ", TT "t", "-spread monomials of the ", TT "t", "-strongly stable set generated by ", TT "u"}},
"the function ", TT "tStronglyStableMon(u,t)", " gives the list of all the monomials belonging to the ", TT "t" , "-strongly stable set generated by ", TT "u", ", that is, ", TEX///$B_t\{u\}.$///,BR{},
"We recall that if ", TEX///$u\in M_{n,d,t}\subset S=K[x_1,\ldots,x_n]$///, " then ", TEX///$B_t\{u\}$///, " is the smallest ",TT "t", "-strongly stable set of monomials of ", TEX///$M_{n,d,t}$///, " containing ", TEX///$u.$///, BR{},
"Moreover, a subset ", TEX///$N\subset M_{n,d,t}$///, " is called a ", TT "t", "-strongly stable set if taking a ", TT "t", "-spread monomial ",TEX///$u\in N$///, ", for all ", TEX///$j\in \mathrm{supp}(u)$///, " and all ", TEX///$i,\ 1\leq i\leq j$///, ", such that ", TEX///$x_i(u/x_j)$///," is a ", TT"t", "-spread monomial, then it follows that ", TEX///$x_i(u/x_j)\in N$///,".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_9]
    tStronglyStableMon(x_2*x_5*x_8,2)
    tStronglyStableMon(x_2*x_5*x_8,3)
///,
SeeAlso =>{isTStronglyStableSeg, tStronglyStableMon},
}

document {
Key => {countTStronglyStableMon,(countTStronglyStableMon,RingElement,ZZ)},
Headline => "give the cardinality of the t-strongly stable set generated by a given monomial",
Usage => "countTStronglyStableMon(u,t)",
Inputs => {"u" => {"a t-spread monomial of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {ZZ => {"the number of all the ", TT "t", "-spread monomials of the ", TT "t", "-strongly stable set generated by ", TT "u"}},
"the function ", TT "countTStronglyStableMon(u,t)", " gives the cardinality of ", TEX///$B_t\{u\}$///, ", the ", TT "t" , "-strongly stable set generated by ", TT "u", ", that is, the number of all the ", TT "t" , "-spread monomials belonging to the smallest ", TT "t", "-strongly stable set containing ", TT "u",BR{},
"This method is not constructive. It uses a theoretical result to obtain the cardinality as the sum of suitable binomial coefficients. The procedure only concerns ", TEX///$\textrm{supp}(\texttt{u}),$///, " that is, the set ", TEX///$\{i_1,i_2,\ldots, i_d\}$///, ", when ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///, " is a ", TEX///$t$///, "-spread monomial.", BR{},
"Moreover, a subset ", TEX///$N\subset M_{n,d,t}$///, " is called a ", TT "t", "-strongly stable set if taking a ", TT "t", "-spread monomial ",TEX///$u\in N$///, ", for all ", TEX///$j\in \mathrm{supp}(u)$///, " and all ", TEX///$i,\ 1\leq i\leq j$///, ", such that ", TEX///$x_i(u/x_j)$///," is a ", TT"t", "-spread monomial, then it follows that ", TEX///$x_i(u/x_j)\in N$///,".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_9]
    countTStronglyStableMon(x_2*x_5*x_8,2)
    countTStronglyStableMon(x_2*x_5*x_8,3)
///,
SeeAlso =>{tStronglyStableMon},
}

document {
Key => {tStronglyStableIdeal,(tStronglyStableIdeal,Ideal,ZZ)},
Headline => "give the smallest t-strongly stable ideal containing a given t-spread ideal",
Usage => "tStronglyStableIdeal(I,t)",
Inputs => {"I" => {"a t-spread ideal of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the smallest ", TT "t", "-strongly stable ideal containing the ", TT "t", "-spread ideal ", TT "I"}},
"the function ", TT "tStronglyStableIdeal(I,t)", " gives the smallest ", TT "t" , "-strongly stable ideal containing ", TT "I", ", that is, ", TEX///$B_t(G(I))$///, " where ", TEX///$G(I)$///, " is the minimal set of generators of ", TEX///$I$///, BR{},
"We recall that if ", TEX///$u\in M_{n,d,t}\subset S=K[x_1,\ldots,x_n]$///, " then ", TEX///$B_t(u)$///, " is the smallest ",TT "t", "-strongly stable ideal of ", TEX///$S$///, " containing ", TEX///$u.$///, BR{},
"Moreover, a subset ", TEX///$N\subset M_{n,d,t}$///, " is called a ", TT "t", "-strongly stable set if taking a ", TT "t", "-spread monomial ",TEX///$u\in N$///, ", for all ", TEX///$j\in \mathrm{supp}(u)$///, " and all ", TEX///$i,\ 1\leq i\leq j$///, ", such that ", TEX///$x_i(u/x_j)$///," is a ", TT"t", "-spread monomial, then it follows that ", TEX///$x_i(u/x_j)\in N$///,".", BR{},
"A ", TT "t", "-spread monomial ideal ", TT "I", " is ", TT "t", "-strongly stable if ", TEX///$[I_j]_t$///, " is a ", TT "t", "-strongly stable set for all ", TEX///$j$///, ", where ", TEX///$[I_j]_t$///, " is the ", TT "t", "-spread part of the ", TEX///$j$///, "-th graded component of ", TT "I", ".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_9]
    tStronglyStableIdeal(ideal {x_2*x_5*x_8},2)
    tStronglyStableIdeal(ideal {x_2*x_5*x_8},3)
///,
SeeAlso =>{isTStronglyStableSeg, tStronglyStableMon},
}

document {
Key => {isTStronglyStableIdeal,(isTStronglyStableIdeal,Ideal,ZZ)},
Headline => "whether a given t-spread ideal is t-strongly stable",
Usage => "isTStronglyStableIdeal(I,t)",
Inputs => {"I" => {"a t-spread ideal of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {Boolean => {"whether the ideal ", TT "I", " is t-spread lex"}},
"Let ", TEX///$\texttt{S}=K[x_1,\ldots,x_n]$///, ", ", TEX///$\texttt{t}\geq 1$///, " and a ", TT "t", "-spread ideal ", TT "I", ". Then ", TT "I", " is called a ", TT "t", "-strongly stable ideal, if ", TEX///$[I_j]_t$///, " is a ", TT "t", "-strongly stable set for all ", TEX///$j$///, ".",BR{},
"We recall that ", TEX///$[I_j]_t$///, " is the ", TT "t", "-spread part of the ", TEX///$j$///, "-th graded component of ", TT "I", "Moreover, a subset ", TEX///$N\subset M_{n,d,t}$///, " is called a ", TT "t", "-strongly stable set if taking a ", TT "t", "-spread monomial ",TEX///$u\in N$///, ", for all ", TEX///$j\in \mathrm{supp}(u)$///, " and all ", TEX///$i,\ 1\leq i\leq j$///, ", such that ", TEX///$x_i(u/x_j)$///," is a ", TT"t", "-spread monomial, then it follows that ", TEX///$x_i(u/x_j)\in N$///,".",
PARA {"Examples:"},
EXAMPLE lines ///
    S=QQ[x_1..x_6]
    isTStronglyStableIdeal(ideal {x_1*x_3,x_1*x_5},2)
    isTStronglyStableIdeal(ideal {x_1*x_3,x_1*x_4,x_1*x_5,x_2*x_4,x_2*x_5},2)
///,
SeeAlso =>{tStronglyStableIdeal},
}

document {
Key => {tExtremalBettiCorners,(tExtremalBettiCorners,Ideal,ZZ)},
Headline => "give the corners of the extremal Betti numbers of a given t-strongly stable ideal",
Usage => "tExtremalBettiCorners(I,t)",
Inputs => {"I" => {"a t-strongly stable graded ideal of a polynomial ring"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the corners of the extremal Betti numbers of the ideal ", TT "I"}},
"let ", TEX///$S=K[x_1,\ldots,x_n]$///, " be a polynomial ring over a field ", TEX///$K$///," and let ", TT "I", " a graded ideal of ", TEX///$S$///,". A graded Betti number of the ideal ", TEX///$\beta_{k,k+\ell}(I)\ne 0$///, " is called extremal if ", TEX///$\beta_{i,i+j}(I)=0$///, " for all ", TEX///$i\ge k,j\ge\ell,(i,j)\ne(k,\ell).$///,BR{},
"Let ", TEX///$\beta_{k,k+\ell}(I)$///, " be an extremal Betti number of ", TT "I", ", the pair ", TEX///$(k,\ell)$///, " is called a corner of ", TT "I", ". If ", TEX///$(k_1,\ell_1),\dots,(k_r,\ell_r)$///, ", with ", TEX///$n-1\ge k_1>k_2>\dots>k_r\ge1 \text{ and } 1\le \ell_1<\ell_2<\dots<\ell_r$///, ", are all the corners of a graded ideal ", TT"I", " of ", TEX///$S$///, ", the set ", TEX///$\mathrm{Corn}(I)=\Big\{(k_1,\ell_1),(k_2,\ell_2),\dots,(k_r,\ell_r)\Big\}$///, " is called the corner sequence of ", TT"I", ".",BR{},
"We recall that ", TT "I", " has a minimal graded free ", TEX///$S$///," resolution:", TEX///$ F_{\scriptscriptstyle\bullet}:0\rightarrow \bigoplus_{j\in\mathbb{Z}}S(-j)^{\beta_{r,j}}\rightarrow \cdots\rightarrow \bigoplus_{j\in\mathbb{Z}}S(-j)^{\beta_{1,j}}\rightarrow \bigoplus_{j\in\mathbb{Z}}S(-j)^{\beta_{0,j}}\rightarrow I\rightarrow 0.$///,BR{},
"The integer ", TEX///$\beta_{i,j}$///, " is a graded Betti number of ", TT "I", " and it represents the dimension as a ", TEX///$K$///, "-vector space of the ", TEX///$j$///, "-th graded component of the ", TEX///$i$///, "-th free module of the resolution. Each of the numbers ", TEX///$\beta_i=\sum_{j\in\mathbb{Z}}\beta_{i,j}$///, " is called the ", TEX///$i$///, "-th Betti number of ", TT "I", ".",
PARA {"Example:"},
EXAMPLE lines ///
    S=QQ[x_1..x_10]
    I=ideal(x_1*x_3,x_1*x_4,x_1*x_5,x_2*x_4*x_6,x_2*x_4*x_7,x_2*x_5*x_7*x_9,x_3*x_5*x_7*x_9)
    tExtremalBettiCorners(I,2)
    minimalBettiNumbersIdeal I
///
}

document {
Key => {tExtremalBettiMonomials,(tExtremalBettiMonomials,Ring,List,List,ZZ)},
Headline => "give the list of the t-spread basic monomials related to the given extremal Betti numbers configuration",
Usage => "tExtremalBettiMonomials(S,corners,a,t)",
Inputs => {"S" => {"a polynomial ring"},
           "corners" => {"a list of pairs of positive integers"},
           "a" => {"a list of positive integers"},
           "t" => {"a positive integer that idenfies the t-spread contest"}
},
Outputs => {List => {"the t-spread basic monomials to obtain the extremal Betti numbers positioned in ", TT "cornes", " and with values ", TT "a"}},
"let ", TEX///$S=K[x_1,\ldots,x_n]$///, " be a polynomial ring over a field ", TEX///$K$///," and let ", TT "I", " a graded ideal of ", TEX///$S$///,". A graded Betti number of the ideal ", TEX///$\beta_{k,k+\ell}(I)\ne 0$///, " is called extremal if ", TEX///$\beta_{i,i+j}(I)=0$///, " for all ", TEX///$i\ge k,j\ge\ell,(i,j)\ne(k,\ell).$///,BR{},
"The basic monomials related to an extremal Betti number configuration (values as well as positions) are defined by the following characterization. Let ", TEX///$I$///, " be a ", TEX///$t$///, "-spread strongly stable ideal of ", TT "S", " and let ", TEX///$\beta_{k,k+\ell}(I)$///, " be an extremal Betti number of ", TEX///$I$///, ". Then ", TEX///$\beta_{k,k+\ell}(I)=\Big|\Big\{ u\in G(I)_\ell:\max(u)=k+t(\ell-1)+1 \Big\}\Big|.$///, BR{},
"Let ", TEX///$\beta_{k,k+\ell}(I)$///, " be an extremal Betti number of ", TT "I", ", the pair ", TEX///$(k,\ell)$///, " is called a corner of ", TT "I", ". If ", TEX///$(k_1,\ell_1),\dots,(k_r,\ell_r)$///, ", with ", TEX///$n-1\ge k_1>k_2>\dots>k_r\ge1 \text{ and } 1\le \ell_1<\ell_2<\dots<\ell_r$///, ", are all the corners of a graded ideal ", TT"I", " of ", TEX///$S$///, ", the set ", TEX///$\mathrm{Corn}(I)=\Big\{(k_1,\ell_1),(k_2,\ell_2),\dots,(k_r,\ell_r)\Big\}$///, " is called the corner sequence of ", TT"I", ".",BR{},
"We recall that ", TT "I", " has a minimal graded free ", TEX///$S$///," resolution:", TEX///$ F_{\scriptscriptstyle\bullet}:0\rightarrow \bigoplus_{j\in\mathbb{Z}}S(-j)^{\beta_{r,j}}\rightarrow \cdots\rightarrow \bigoplus_{j\in\mathbb{Z}}S(-j)^{\beta_{1,j}}\rightarrow \bigoplus_{j\in\mathbb{Z}}S(-j)^{\beta_{0,j}}\rightarrow I\rightarrow 0.$///,BR{},
"The integer ", TEX///$\beta_{i,j}$///, " is a graded Betti number of ", TT "I", " and it represents the dimension as a ", TEX///$K$///, "-vector space of the ", TEX///$j$///, "-th graded component of the ", TEX///$i$///, "-th free module of the resolution. Each of the numbers ", TEX///$\beta_i=\sum_{j\in\mathbb{Z}}\beta_{i,j}$///, " is called the ", TEX///$i$///, "-th Betti number of ", TT "I", ".",
PARA {"Example:"},
EXAMPLE lines ///
    S=QQ[x_1..x_12]
    corners={{4,2},{3,4},{2,5}}
    a={1,2,1}
    l=tExtremalBettiMonomials(S,corners,a,2)
    I=tStronglyStableIdeal(ideal l,2)
    minimalBettiNumbersIdeal I
///
}

document {
Key => {solveBinomialExpansion,(solveBinomialExpansion,List)},
Headline => "compute the sum of a binomial expansion",
Usage => "solveBinomialExpansion l",
Inputs => {"l" => {"a list of pairs of integers representing a binomial expansion"}
},
Outputs => {ZZ => {"the sum of binomials in the list", TT "l"}},
"Given a list of pairs ", TEX///$\{\{a_1,b_1\}, ... ,\{a_k,b_k\}\}$///, " this method yields the sum of binomials ", TEX///$\binom{a_1,b_1} + \cdots + \binom{a_k,b_k}.$///,
PARA {"Example:"},
EXAMPLE lines ///
    solveBinomialExpansion({{4,2},{3,1}})
    solveBinomialExpansion tMacaulayExpansion(50,12,2,1)
///,
SeeAlso =>{tMacaulayExpansion}
}

document {
Key => {initialDegree,(initialDegree,Ideal)},
Headline => "return the initial degree of a given graded ideal",
Usage => "initialDegree I",
Inputs => {"I" => {"a graded ideal of a polynomial ring"}
},
Outputs => {ZZ => {"the initial degree of the ideal ", TT "I"}},
"The initial degree of a graded ideal ", TT "I", " is the least degree of a homogeneous generator of ", TT "I", ".",
PARA {"Example:"},
EXAMPLE lines ///
    S=QQ[x_1..x_4]
    initialDegree ideal {x_1*x_2,x_2*x_3*x_4}
    initialDegree ideal {x_1*x_3*x_4}
///
}

document {
Key => {initialIdeal,(initialIdeal,Ideal)},
Headline => "return the initial ideal of a given ideal",
Usage => "initialIdeal I",
Inputs => {"I" => {"a graded ideal of a polynomial ring"}
},
Outputs => {Ideal => {"the initial ideal of the ideal ", TT "I", " with default monomial order"}},
"let ", TEX///$S=K[x_1,\ldots,x_n]$///, " be a polynomial ring over a field ", TEX///$K$///, ". Let > be a monomial order on ", TEX///$S$///, ". The largest monomial of a polynomial ", TEX///$f\in S$///, " is called the initial monomial of ", TEX///$f$///, " and it is denoted by ", TEX///$\mathrm{In}(f).$///,BR{},
"If ", TT "I", " is a graded ideal of ", TEX///$S$///, " then the initial ideal of ", TT "I", ", denoted by ", TEX///$\mathrm{In}(I)$///, ", is the ideal of ", TEX///$S$///, " generated by the initial terms of elements of ", TT "I", ".",
PARA {"Example:"},
EXAMPLE lines ///
    S=QQ[x_1..x_5]
    I=ideal {x_1*x_2+x_3*x_4*x_5,x_1*x_3+x_4*x_5,x_2*x_3*x_4}
    initialIdeal I
///
}

document {
Key => {minimalBettiNumbers,(minimalBettiNumbers,Ideal)},
Headline => "return the minimal Betti numbers of the quotient ring corresponding to a given graded ideal",
Usage => "minimalBettiNumbers I",
Inputs => {"I" => {"a graded ideal of a polynomial ring"}
},
Outputs => {BettiTally => {"the Betti table of the quotient ", TT "S/I", " computed using its minimal generators"}},
PARA {"Example:"},
EXAMPLE lines ///
    S=QQ[x_1..x_4]
    I=ideal(x_1*x_2,x_1*x_3,x_2*x_3)
    J=ideal(join(flatten entries gens I,{x_1*x_2*x_3}))
    I==J
    betti I==betti J
    minimalBettiNumbers I==minimalBettiNumbers J
///
}

document {
Key => {minimalBettiNumbersIdeal,(minimalBettiNumbersIdeal,Ideal)},
Headline => "return the minimal Betti numbers of a given graded ideal",
Usage => "minimalBettiNumbersIdeal I",
Inputs => {"I" => {"a graded ideal of a polynomial ring"}
},
Outputs => {BettiTally => {"the Betti table of the ideal ", TT "I", " computed using its minimal generators"}},
"let ", TEX///$S=K[x_1,\ldots,x_n]$///, " be a polynomial ring over a field ", TEX///$K$///," and let ", TT "I", " a graded ideal of ", TEX///$S$///,". Then ", TT "I", " has a minimal graded free ", TEX///$S$///," resolution:", TEX///$ F_{\scriptscriptstyle\bullet}:0\rightarrow \bigoplus_{j\in\mathbb{Z}}S(-j)^{\beta_{r,j}}\rightarrow \cdots\rightarrow \bigoplus_{j\in\mathbb{Z}}S(-j)^{\beta_{1,j}}\rightarrow \bigoplus_{j\in\mathbb{Z}}S(-j)^{\beta_{0,j}}\rightarrow I\rightarrow 0.$///,BR{},
"The integer ", TEX///$\beta_{i,j}$///, " is a graded Betti number of ", TT "I", " and it represents the dimension as a ", TEX///$K$///, "-vector space of the ", TEX///$j$///, "-th graded component of the ", TEX///$i$///, "-th free module of the resolution. Each of the numbers ", TEX///$\beta_i=\sum_{j\in\mathbb{Z}}\beta_{i,j}$///, " is called the ", TEX///$i$///, "-th Betti number of ", TT "I", ".",
PARA {"Example:"},
EXAMPLE lines ///
    S=QQ[x_1..x_4]
    I=ideal(x_1*x_2,x_1*x_3,x_2*x_3)
    J=ideal(join(flatten entries gens I,{x_1*x_2*x_3}))
    I==J
    betti I==betti J
    minimalBettiNumbersIdeal I==minimalBettiNumbersIdeal J
///
}

------------------------------------------------------------
-- DOCUMENTATION FOR OPTIONS
------------------------------------------------------------

----------------------------------
-- Shift (for tMacaulayExpansion)
----------------------------------

document {
Key => {Shift,[tMacaulayExpansion,Shift]},
Headline => "optional boolean argument for tMacaulayExpansion",
"whether it is true the function tMacaulayExpansion gives the ", TEX///$i$///, "-th shifted Macaulay expansion of ", TEX///$a$///, ". Given four positive integers ", TEX///$(a,n,d,t)$///, " the ", TEX///$d$///,"-th shifted t-Macaulay expansion is a sum of binomials: ", TEX///$\binom{a_d}{d+1} + \binom{a_{d-1}}{d} + \cdots + \binom{a_j}{j+1}.$///,
SeeAlso =>{tMacaulayExpansion}
}

----------------------------------
-- FixedMax (for tNextMon and countTLexMon)
----------------------------------

document {
Key => {FixedMax,[tNextMon,FixedMax],[countTLexMon,FixedMax]},
Headline => "optional boolean argument for tNextMon",
"whether it is ", TT "true", " the function ", TT "tNextMon(u,t,FixedMax=>true)", " gives the ", TT "t" , "-lex successor for which the maximum of the support is ", TEX///$\max\textrm{supp}(\texttt{u})$///, ". Given a ", TEX///$t$///, "-spread monomial ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///,", we define ", TEX///$\textrm{supp}(u)=\{i_1,i_2,\ldots, i_d\}$///, ".",
SeeAlso =>{tNextMon, countTLexMon}
}

----------------------------------
-- MaxInd (for tLastMon and countTLexMon)
----------------------------------

document {
Key => {MaxInd,[tLastMon,MaxInd]},
Headline => "optional integer argument for tLastMon",
"whether it is greater than -1 the function ", TT "tLastMon(u,t,MaxInd=>m)", " gives the smallest ", TT "t", "-spread monomial for which the maximum of the support is ", TEX///$\max\textrm{supp}(\texttt{u})$///, " of the Borel shadow of ", TEX///$B_\texttt{t}\{\texttt{u}\}$///, ". Given a ", TEX///$t$///, "-spread monomial ", TEX///$u=x_{i_1}x_{i_2}\cdots x_{i_d}$///,", we define ", TEX///$\textrm{supp}(u)=\{i_1,i_2,\ldots, i_d\}$///, ".",
SeeAlso =>{tLastMon, tShadow}
}

------------------------------------------------------------
-- TESTS
------------------------------------------------------------

----------------------------
-- Test 0 tNextMon
----------------------------
TEST ///
S=QQ[x_1..x_16]
assert(tNextMon(x_2*x_6*x_10*x_13,3)==x_2*x_6*x_10*x_14)
assert(tNextMon(x_2*x_6*x_10*x_13,3,FixedMax=>true)==x_2*x_7*x_10*x_13)
///

----------------------------
-- Test 1 tLastMon
----------------------------
TEST ///
S=QQ[x_1..x_16]
assert(tLastMon(x_2*x_6*x_10*x_13,1,3)==x_2*x_6*x_10*x_13*x_16)
assert(tLastMon(x_2*x_6*x_10*x_13,1,3,MaxInd=>14)==x_2*x_5*x_8*x_11*x_14)
///

----------------------------
-- Test 2 tLexSeg
----------------------------
TEST ///
S=QQ[x_1..x_14]
assert(tLexSeg(x_2*x_5*x_10*x_13,x_2*x_6*x_9*x_12,3)=={x_2*x_5*x_10*x_13,x_2*x_5*x_10*x_14,x_2*x_5*x_11*x_14,x_2*x_6*x_9*x_12})
///

----------------------------
-- Test 3 isTLexSeg
----------------------------
TEST ///
S=QQ[x_1..x_7]
assert(isTLexSeg({x_1*x_4*x_7,x_1*x_5*x_7,x_2*x_4*x_6,x_2*x_4*x_7,x_2*x_5*x_7},2))
assert(not isTLexSeg({x_1*x_4*x_7,x_2*x_4*x_6,x_2*x_4*x_7,x_2*x_5*x_7},2))
///

----------------------------
-- Test 4 tLexMon
----------------------------
TEST ///
S=QQ[x_1..x_9]
assert(tLexMon(x_2*x_5*x_8,3)=={x_1*x_4*x_7,x_1*x_4*x_8,x_1*x_4*x_9,x_1*x_5*x_8,x_1*x_5*x_9,x_1*x_6*x_9,x_2*x_5*x_8})
///

----------------------------
-- Test 5 countTLexMon
----------------------------
TEST ///
S=QQ[x_1..x_11]
assert(countTLexMon(x_2*x_6*x_10,2)==42)
assert(countTLexMon(x_2*x_6*x_10,3)==21)
///

----------------------------
-- Test 6 tVeroneseSet
----------------------------
TEST ///
S=QQ[x_1..x_8]
assert(tVeroneseSet(S,3,3)=={x_1*x_4*x_7,x_1*x_4*x_8,x_1*x_5*x_8,x_2*x_5*x_8})
///

----------------------------
-- Test 7 tVeroneseIdeal
----------------------------
TEST ///
S=QQ[x_1..x_8]
assert(tVeroneseIdeal(S,3,3)==ideal {x_1*x_4*x_7,x_1*x_4*x_8,x_1*x_5*x_8,x_2*x_5*x_8})
assert(tVeroneseIdeal(S,3,4)==ideal {0_S})
///

----------------------------
-- Test 8 tPascalIdeal
----------------------------
TEST ///
S=QQ[x_1..x_10]
assert(tPascalIdeal(S,3)==ideal {x_1*x_4*x_7*x_10,x_2*x_5*x_8,x_3*x_6*x_9})
assert(tPascalIdeal(S,4)==ideal {x_1*x_5*x_9,x_2*x_6*x_10,x_3*x_7,x_4*x_8})
///

----------------------------
-- Test 9 tSpreadList
----------------------------
TEST ///
S=QQ[x_1..x_14]
l={x_3*x_7*x_10*x_14, x_1*x_5*x_9*x_13}
assert(tSpreadList(l,4)== {x_1*x_5*x_9*x_13})
assert(tSpreadList(l,5)== {})
///

----------------------------
-- Test 10 tSpreadIdeal
----------------------------
TEST ///
S=QQ[x_1..x_14]
I=ideal {x_3*x_7*x_10*x_14, x_1*x_5*x_9*x_13}
assert(tSpreadIdeal(I,3)== I)
assert(tSpreadIdeal(I,4)== ideal {x_1*x_5*x_9*x_13})
///

----------------------------
-- Test 11 isTSpread
----------------------------
TEST ///
S=QQ[x_1..x_14]
l={x_3*x_7*x_10*x_14, x_1*x_5*x_9*x_13}
assert(isTSpread(l#0,3))
assert(isTSpread(l,3))
assert(isTSpread(ideal l,3))
assert(not isTSpread(ideal l,4))
///

----------------------------
-- Test 12 tShadow
----------------------------
TEST ///
S=QQ[x_1..x_14]
u=x_2*x_6*x_10
assert(tShadow(u,3)=={x_2*x_6*x_10*x_13, x_2*x_6*x_10*x_14})
assert(tShadow(u,4)=={x_2*x_6*x_10*x_14})
l={x_3*x_6*x_10, x_1*x_5*x_9}
assert(tShadow(l,3)=={x_1*x_5*x_9*x_12,x_1*x_5*x_9*x_13,x_1*x_5*x_9*x_14,x_3*x_6*x_10*x_13, x_3*x_6*x_10*x_14})
assert(tShadow(l,4)=={x_1*x_5*x_9*x_13,x_1*x_5*x_9*x_14})
///

----------------------------
-- Test 13 tMacaulayExpansion
----------------------------
TEST ///
assert(tMacaulayExpansion(50,12,2,1)=={{10,2},{5,1}})
assert(tMacaulayExpansion(50,12,2,1,Shift=>true)=={{10,3},{5,2}})
assert(tMacaulayExpansion(50,12,2,2,Shift=>true)=={{9,3},{4,2}})
///

----------------------------
-- Test 14 fTVector
----------------------------
TEST ///
S=QQ[x_1..x_8]
assert(fTVector(ideal {x_1*x_3,x_2*x_5*x_7},1)=={1,8,27,49,50,27,6,0,0})
assert(fTVector(ideal {x_1*x_3,x_2*x_5*x_7},2)=={1,8,20,15,2})
///

----------------------------
-- Test 15 isFTVector
----------------------------
TEST ///
S=QQ[x_1..x_8]
f={1,8,20,10,0}
assert(isFTVector(S,f,2)==true)
S=QQ[x_1..x_7]
assert(isFTVector(S,f,2)==false)
///

----------------------------
-- Test 16 tLexIdeal
----------------------------
TEST ///
S=QQ[x_1..x_8]
f={1,8,2,0,0}
I=tLexIdeal(S,f,2)
assert(fTVector(I,2)==f)
assert(isTLexIdeal(I,2)==true)
J=tStronglyStableIdeal(ideal {x_1*x_4*x_6},2)
K=tLexIdeal(J,2)
assert(fTVector(J,2)==fTVector(K,2))
///

----------------------------
-- Test 17 isTLexIdeal
----------------------------
TEST ///
S=QQ[x_1..x_6]
assert(not isTLexIdeal(ideal {x_1*x_3,x_1*x_5},2))
assert(isTLexIdeal(ideal {x_1*x_3,x_1*x_4,x_1*x_5,x_1*x_6,x_2*x_4,x_2*x_5},2))
///

----------------------------
-- Test 18 tStronglyStableSeg
----------------------------
TEST ///
S=QQ[x_1..x_14]
assert(#tStronglyStableSeg(x_2*x_5*x_9*x_12,x_2*x_6*x_10*x_13,2)==13)
assert(tStronglyStableSeg(x_2*x_5*x_9*x_12,x_2*x_6*x_10*x_13,3)=={x_2*x_5*x_9*x_12,x_2*x_5*x_9*x_13,x_2*x_5*x_10*x_13,x_2*x_6*x_9*x_12,x_2*x_6*x_9*x_13,x_2*x_6*x_10*x_13})
///

----------------------------
-- Test 19 isTStronglyStableSeg
----------------------------
TEST ///
S=QQ[x_1..x_9]
assert(isTStronglyStableSeg({x_1*x_4*x_7,x_1*x_4*x_8,x_1*x_5*x_8,x_2*x_5*x_8},3))
assert(not isTStronglyStableSeg({x_1*x_4*x_7,x_1*x_4*x_8,x_2*x_5*x_8},3))
///

----------------------------
-- Test 20 tStronglyStableMon
----------------------------
TEST ///
S=QQ[x_1..x_9]
assert(#tStronglyStableMon(x_2*x_5*x_8,2)==14)
assert(tStronglyStableMon(x_2*x_5*x_8,3)=={x_1*x_4*x_7,x_1*x_4*x_8,x_1*x_5*x_8,x_2*x_5*x_8})
///

----------------------------
-- Test 21 countTStronglyStableMon
----------------------------
TEST ///
S=QQ[x_1..x_9]
assert(countTStronglyStableMon(x_2*x_5*x_8,2)==14)
assert(countTStronglyStableMon(x_2*x_5*x_8,3)==4)
///

----------------------------
-- Test 22 tStronglyStableIdeal
----------------------------
TEST ///
S=QQ[x_1..x_9]
assert(numgens tStronglyStableIdeal(ideal {x_2*x_5*x_8},2)==14)
assert(tStronglyStableIdeal(ideal {x_2*x_5*x_8},3)==ideal {x_1*x_4*x_7,x_1*x_4*x_8,x_1*x_5*x_8,x_2*x_5*x_8})
///

----------------------------
-- Test 23 isTStronglyStableIdeal
----------------------------
TEST ///
S=QQ[x_1..x_6]
assert(not isTStronglyStableIdeal(ideal {x_1*x_3,x_1*x_5},2))
assert(isTStronglyStableIdeal(ideal {x_1*x_3,x_1*x_4,x_1*x_5,x_2*x_4,x_2*x_5},2))
///

----------------------------
-- Test 24 tExtremalBettiCorners
----------------------------
TEST ///
S=QQ[x_1..x_10]
I=ideal(x_1*x_3,x_1*x_4,x_1*x_5,x_2*x_4*x_6,x_2*x_4*x_7,x_2*x_5*x_7*x_9,x_3*x_5*x_7*x_9)
assert(tExtremalBettiCorners(I,2)=={(2,4)})
J=tStronglyStableIdeal(I,1)
assert(tExtremalBettiCorners(J,1)=={(5,4)})
///

----------------------------
-- Test 25 tExtremalBettiMonomials
----------------------------
TEST ///
S=QQ[x_1..x_12]
corners={{4,2},{3,4},{2,5}}
a={1,2,1}
assert(tExtremalBettiMonomials(S,corners,a,2)=={x_1*x_7,x_2*x_4*x_6*x_10,x_2*x_4*x_7*x_10,x_2*x_5*x_7*x_9*x_11})
///

----------------------------
-- Test 26 solveBinomialExpansion
----------------------------
TEST ///
assert(solveBinomialExpansion({{5,5},{3,4},{2,3},{1,2}})==1)
assert(solveBinomialExpansion({{6,6},{5,5},{4,4},{3,3},{2,2}})==5)
assert(solveBinomialExpansion tMacaulayExpansion(50,12,2,1)==50)
///

----------------------------
-- Test 27 initialDegree
----------------------------
TEST ///
S=QQ[x_1..x_4]
assert(initialDegree ideal {x_1*x_2,x_1*x_2*x_3}==2)
assert(initialDegree ideal {x_1*x_2*x_3}==3)
///

----------------------------
-- Test 28 initialIdeal
----------------------------
TEST ///
S=QQ[x_1..x_5]
I=ideal {x_1*x_2+x_3*x_4*x_5,x_1*x_3+x_4*x_5}
J=ideal {x_1*x_3,x_3*x_4*x_5,x_4^2*x_5^2}
assert(initialIdeal I==J)
///

----------------------------
-- Test 29 minimalBettiNumbers
----------------------------
TEST ///
S=QQ[x_1..x_4]
I=ideal {x_1*x_2,x_1*x_3,x_2*x_3}
J=ideal(join(flatten entries gens I,{x_1*x_2*x_3}))
assert(I==J)
assert(minimalBettiNumbers I==minimalBettiNumbers J)
///

----------------------------
-- Test 30 minimalBettiNumbersIdeal
----------------------------
TEST ///
S=QQ[x_1..x_4]
I=ideal {x_1*x_2,x_1*x_3,x_2*x_3}
J=ideal(join(flatten entries gens I,{x_1*x_2*x_3}))
assert(I==J)
assert(minimalBettiNumbersIdeal I==minimalBettiNumbersIdeal J)
///

end

restart
installPackage ("TSpreadIdeals", UserMode=>true, RerunExamples=>true)
loadPackage "TSpreadIdeals"
viewHelp