/**
 *    Copyright 2011 Peter Murray-Rust et. al.
 *
 *    Licensed under the Apache License, Version 2.0 (the "License");
 *    you may not use this file except in compliance with the License.
 *    You may obtain a copy of the License at
 *
 *        http://www.apache.org/licenses/LICENSE-2.0
 *
 *    Unless required by applicable law or agreed to in writing, software
 *    distributed under the License is distributed on an "AS IS" BASIS,
 *    WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *    See the License for the specific language governing permissions and
 *    limitations under the License.
 */

package org.xmlcml.cml.element;

import java.util.ArrayList;
import java.util.List;

import nu.xom.Element;
import nu.xom.Elements;
import org.xmlcml.cml.base.CMLElement;
import org.xmlcml.cml.base.CMLElements;

/**
 * user-modifiable class supporting parameterList. * autogenerated from schema
 * use as a shell which can be edited
 *
 */
public class CMLParameterList extends AbstractParameterList {

	/** namespaced element name.*/
	public final static String NS = C_E+TAG;

    /**
     * constructor.
     */
    public CMLParameterList() {
    }

    /**
     * constructor.
     *
     * @param old
     */
    public CMLParameterList(CMLParameterList old) {
        super((AbstractParameterList) old);

    }

    /**
     * copy node .
     *
     * @return Node
     */
    public Element copy() {
        return new CMLParameterList(this);

    }

    /**
     * create new instance in context of parent, overridable by subclasses.
     *
     * @param parent
     *            parent of element to be constructed (ignored by default)
     * @return CMLParameterList
     */
    public CMLElement makeElementInContext(Element parent) {
        return new CMLParameterList();

    }

    /**
     * gets all nested parameter descendants of arbitrary CMLElement.
     *
     * recurses through all parameterList and parameter children, adding the
     * latter to the list. Order is not preserved.
     *
     * @return list of all descendants
     */
    public List<CMLParameter> getParameterDescendants() {
        List<CMLParameter> mList = new ArrayList<CMLParameter>();
        CMLElements<CMLParameterList> parameterLists = this
                .getParameterListElements();
        for (CMLParameterList ml : parameterLists) {
            mList.addAll(ml.getParameterDescendants());
        }
        CMLElements<CMLParameter> parameters = this.getParameterElements();
        for (CMLParameter m : parameters) {
            mList.add(m);
        }
        return mList;
    }

    /**
     * gets all nested parameter descendants of arbitrary CMLElement.
     *
     * recurses through all parameterList and parameter children, adding the
     * latter to the list. Order is not preserved.
     *
     * @param parent with parameter(List) children
     * @return list of all descendants
     */
    public static List<CMLParameter> getParameterDescendants(CMLElement parent) {
        List<CMLParameter> mList = new ArrayList<CMLParameter>();
        Elements parameterLists = parent
                .getChildCMLElements(CMLParameterList.TAG);
        for (int i = 0; i < parameterLists.size(); i++) {
            CMLParameterList parameterList = (CMLParameterList) parameterLists
                    .get(i);
            mList.addAll(parameterList.getParameterDescendants());
        }
        // generally parameter should be under parameterList parent
        Elements parameters = parent.getChildCMLElements(CMLParameter.TAG);
        for (int i = 0; i < parameters.size(); i++) {
            mList.add((CMLParameter) parameters.get(i));
        }
        return mList;
    }

    /**
     * gets parameter elements with a given dictRef. if either param is null,
     * returns empty list
     *
     * @param parameterList list to filter
     * @param dictRef value of dictRef attribute
     * @return filtered list
     */
    public static List<CMLParameter> getParameterDescendantsByDictRef(
            List<CMLParameter> parameterList, String dictRef) {
        List<CMLParameter> newParameterList = new ArrayList<CMLParameter>();
        if (dictRef != null && parameterList != null) {
            for (CMLParameter parameter : parameterList) {
                if (dictRef.equals(parameter.getDictRef())) {
                    newParameterList.add(parameter);
                }
            }
        }
        return newParameterList;
    }

    /**
     * gets parameter elements with a given name attribute.
     *
     * @param name (if null returns null)
     * @return filtered list or null
     */
    public List<CMLParameter> getParameterDescendantsByName(String name) {
        List<CMLParameter> parameterList = this.getParameterDescendants();
        List<CMLParameter> newParameterList = new ArrayList<CMLParameter>();
        if (name != null && parameterList != null) {
            for (CMLParameter parameter : parameterList) {
                if (name.equals(parameter.getName())) {
                    newParameterList.add(parameter);
                }
            }
        }
        return newParameterList;
    }

}
