(* :Name: Delaunay *)

(* :Title: Delaunay *)

(* :Author: Tom Wickham-Jones*)

(* :Package Version: 1.0 *)

(* :Mathematica Version: 2.2 *)

(*:Summary:
	This package provides functions to generate delaunay triangulations
	in the plane.

*)

(* :History:
	Created summer 1993 by Tom Wickham-Jones.
	New version winter 1993 by Tom Wickham-Jones.

	This package is described in the book
	Mathematica Graphics: Techniques and Applications.
	Tom Wickham-Jones, TELOS/Springer-Verlag 1994.

*)

(*:Warnings:
	The package installs the MathLink binary delaunay.
	If this fails the functions cannot possibly work.
	If the installation fails an error message is printed.
	In case of a failure make sure that the binary is built
	properly for your system.	

	Removes ConvexHull and ConvexHullPlot in case they
	have already been defined.
*)

If[ Context[ ConvexHull] =!= "ExtendGraphics`Delaunay`",
		Remove[ ConvexHull]]

If[ Context[ ConvexHullPlot] =!= "ExtendGraphics`Delaunay`",
		Remove[ ConvexHullPlot]]


BeginPackage[ "ExtendGraphics`Delaunay`", "Utilities`FilterOptions`"]

Delaunay::usage = 
	"Delaunay[ pts] will take a list of points.  Each
	point of the form {x, y, ...}.   The Delaunay triangulation 
	in the xy plane will be made.  It will return {hull, adj, tri} 
	where hull is the convex hull of the points, adj is the adjacency
	matrix and tris are the triangles. Each element of
	the adjacency matrix is of the form {e, a, b, c, ...}, indicating
	that the e'th point is connected to the a'th, b'th, c'th etc..
	points."


TrianglePlot::usage = 
	"TrianglePlot[ pts, opts] will take a list of points.
	Each point of the form {x, y}. It will plot the triangles
	which form the Delaunay triangulation."


TriangulatePoints::usage = 
	"TriangulatePoints[ pts] will take a list of
	points, {x, y, z1, ...}, and return the list of triples 
	of points which corresponds to the Delaunay triangulation 
	of the points in the xy plane."


DelaunayVertices::usage = 
	"DelaunayVertices[ pts] will take a list
	of points, {x, y, ...}, and return the list of vertices which forms
	the Delaunay triangulation in the xy plane."


ConvexHull::usage = 
	"ConvexHull[ pts] will take a list of points
	and return the vertices of the convex hull in the xy plane."
	
ConvexHullPlot::usage = 
	"ConvexHullPlot[ pts] will plot the convex hull of a list of 
	two-dimensonal points."


Begin["`Private`"]

Delaunay::fail = 
	"Installation of the MathLink binary delaunay failed.
	 The binary must be found for the functions to work."




If[ Install[ "ExtendGraphics`delaunay`"] === $Failed,
		Message[ Delaunay::fail]]


Delaunay[  pts_List /; MatrixQ[ N[pts], NumberQ] && 
		        Length[ First[ pts]] === 2] :=
	DelaunayL[ N[pts]]


Delaunay[ pts_List /; MatrixQ[ N[pts], NumberQ]] :=
	Delaunay[ Map[ Take[#, 2]&, pts]]


TriangulatePoints[ pts_List /; 
			MatrixQ[ N[pts], NumberQ] && 
		       	Length[ First[ pts]] > 1] :=
    Block[{npts, tri},
	npts = Map[ Take[#, 2]&, pts] ;
	tri = Last[ Delaunay[ npts]] ;
	Map[ Part[ pts, #]&, tri]
	] 

	
ConvexHull[ pts_List /; 
			MatrixQ[ N[pts], NumberQ] && 
		        Length[ First[ pts]] > 1] :=
	First[ Delaunay[ Map[ Take[ #, 2]&, pts]]]


DelaunayVertices[ pts_List /; 
			MatrixQ[ N[pts], NumberQ] && 
		        Length[ First[ pts]] > 1] :=
	Last[ Delaunay[ Map[ Take[ #, 2]&, pts]]]


TrianglePlot[ pts_List /; 
			MatrixQ[ N[pts], NumberQ] && 
		        Length[ First[ pts]] === 2,
		opts___] :=
    Block[{hull, tri, pt, ht},
    	{hull, tri} = Delaunay[ pts] ;
		ht = Map[ Part[ pts, #]&, tri] ;
		pt = Map[ Point[ Apply[ Plus, #]/3.]&, ht] ;
		ht = {Thickness[ 0.001], 
			Map[ Line[ # /. {x_, y__} :> {x,y,x}]&,
		     	ht]} ;
		Show[ Graphics[ ht, opts,
				AspectRatio -> Automatic,
				PlotRange -> All]]
	]

ConvexHullPlot[ pts_List /; 
			MatrixQ[ pts, NumberQ] && 
		        Length[ First[ pts]] === 2, opts___] :=
	Block[{hull},
		hull = ConvexHull[ pts] ;
		hull = Part[ pts, hull] ;
		hull = Line[ Append[ hull, First[ hull]]] ;
		Show[ Graphics[ {hull, AbsolutePointSize[ 3], 
				Map[ Point, pts]}, AspectRatio -> Automatic], opts]
		]

End[]

EndPackage[]

(*:Examples:
	
<<ExtendGraphics/Delaunay.m

d1 = {{0,0}, {3,3}, {6,5}, {9,1}}

Delaunay[ d1]

TrianglePlot[ d1]

d2 = Table[ { x = Random[], y = Random[], x y}, {30}];

Delaunay[ d2]

TrianglePlot[ Map[ Take[ #, 2]&, d2]]

ConvexHull[ d2]

DelaunayAdjacency[ d2]

DelaunayVertices[ d2]

TriangulatePoints[ d2]

<<Graphics/Animation.m

d3 = Table[ {Random[], Random[]}, {30}];

d3 = Sort[ d3, 
	      (py = Floor[ 10 Part[#1, 2]];
               qy = Floor[ 10 Part[#2, 2]];
		  If[ py === qy,
	 	    	Part[#1, 1] < Part[ #2, 1],
			py < qy])&] ;

l = Table[ 
	TrianglePlot[ 
		Take[ d3, i],
      		PlotRange -> {{0,1}, {0,1}},
      		AspectRatio -> 1], {i,5,Length[ d3]}];

ShowAnimation[l]


*)




