/*
 * Copyright (C) 2020 The Android Open Source Project
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package android.view.translation;

import android.annotation.NonNull;
import android.icu.util.ULocale;
import android.os.Parcel;
import android.os.Parcelable;

import com.android.internal.util.DataClass;

import java.util.Objects;

/**
 * Specs and additional info for the translation data.
 *
 * <p>This spec help specify information such as the language/locale for the translation, as well
 * as the data format for the translation (text, audio, etc.)</p>
 */
@DataClass(genEqualsHashCode = true, genHiddenConstDefs = true, genToString = true,
        genConstructor = false)
public final class TranslationSpec implements Parcelable {

    /** Data format for translation is text. */
    public static final int DATA_FORMAT_TEXT = 1;

    /** @hide */
    @android.annotation.IntDef(prefix = "DATA_FORMAT_", value = {
            DATA_FORMAT_TEXT
    })
    @java.lang.annotation.Retention(java.lang.annotation.RetentionPolicy.SOURCE)
    @DataClass.Generated.Member
    public @interface DataFormat {}

    /**
     * @removed use {@code mLocale} instead.
     */
    @Deprecated
    private final @NonNull String mLanguage;

    /**
     * {@link ULocale} representing locale information of this spec.
     */
    private final @NonNull ULocale mLocale;

    private final @DataFormat int mDataFormat;

    void parcelLocale(Parcel dest, int flags) {
        dest.writeSerializable(mLocale);
    }

    static ULocale unparcelLocale(Parcel in) {
        return (ULocale) in.readSerializable(android.icu.util.ULocale.class.getClassLoader(), android.icu.util.ULocale.class);
    }

    /**
     * @removed use {@link #TranslationSpec(ULocale, int)} instead.
     */
    @Deprecated
    public TranslationSpec(@NonNull String language, @DataFormat int dataFormat) {
        mLanguage = language;
        mDataFormat = dataFormat;
        mLocale = new ULocale.Builder().setLanguage(language).build();
    }

    /**
     * Constructs a translation spec with the given locale and data format.
     *
     * @param locale locale of the associated translation data.
     * @param dataFormat data format of the associated translation data.
     */
    public TranslationSpec(@NonNull ULocale locale, @DataFormat int dataFormat) {
        Objects.requireNonNull(locale);
        mLanguage = locale.getLanguage();
        mLocale = locale;
        mDataFormat = dataFormat;
    }



    // Code below generated by codegen v1.0.23.
    //
    // DO NOT MODIFY!
    // CHECKSTYLE:OFF Generated code
    //
    // To regenerate run:
    // $ codegen $ANDROID_BUILD_TOP/frameworks/base/core/java/android/view/translation/TranslationSpec.java
    //
    // To exclude the generated code from IntelliJ auto-formatting enable (one-time):
    //   Settings > Editor > Code Style > Formatter Control
    //@formatter:off


    /**
     * @removed use {@code mLocale} instead.
     */
    @DataClass.Generated.Member
    public @Deprecated @NonNull String getLanguage() {
        return mLanguage;
    }

    /**
     * {@link ULocale} representing locale information of this spec.
     */
    @DataClass.Generated.Member
    public @NonNull ULocale getLocale() {
        return mLocale;
    }

    @DataClass.Generated.Member
    public @DataFormat int getDataFormat() {
        return mDataFormat;
    }

    @Override
    @DataClass.Generated.Member
    public String toString() {
        // You can override field toString logic by defining methods like:
        // String fieldNameToString() { ... }

        return "TranslationSpec { " +
                "language = " + mLanguage + ", " +
                "locale = " + mLocale + ", " +
                "dataFormat = " + mDataFormat +
        " }";
    }

    @Override
    @DataClass.Generated.Member
    public boolean equals(@android.annotation.Nullable Object o) {
        // You can override field equality logic by defining either of the methods like:
        // boolean fieldNameEquals(TranslationSpec other) { ... }
        // boolean fieldNameEquals(FieldType otherValue) { ... }

        if (this == o) return true;
        if (o == null || getClass() != o.getClass()) return false;
        @SuppressWarnings("unchecked")
        TranslationSpec that = (TranslationSpec) o;
        //noinspection PointlessBooleanExpression
        return true
                && Objects.equals(mLanguage, that.mLanguage)
                && Objects.equals(mLocale, that.mLocale)
                && mDataFormat == that.mDataFormat;
    }

    @Override
    @DataClass.Generated.Member
    public int hashCode() {
        // You can override field hashCode logic by defining methods like:
        // int fieldNameHashCode() { ... }

        int _hash = 1;
        _hash = 31 * _hash + Objects.hashCode(mLanguage);
        _hash = 31 * _hash + Objects.hashCode(mLocale);
        _hash = 31 * _hash + mDataFormat;
        return _hash;
    }

    @Override
    @DataClass.Generated.Member
    public void writeToParcel(@NonNull Parcel dest, int flags) {
        // You can override field parcelling by defining methods like:
        // void parcelFieldName(Parcel dest, int flags) { ... }

        dest.writeString(mLanguage);
        parcelLocale(dest, flags);
        dest.writeInt(mDataFormat);
    }

    @Override
    @DataClass.Generated.Member
    public int describeContents() { return 0; }

    /** @hide */
    @SuppressWarnings({"unchecked", "RedundantCast"})
    @DataClass.Generated.Member
    /* package-private */ TranslationSpec(@NonNull Parcel in) {
        // You can override field unparcelling by defining methods like:
        // static FieldType unparcelFieldName(Parcel in) { ... }

        String language = in.readString();
        ULocale locale = unparcelLocale(in);
        int dataFormat = in.readInt();

        this.mLanguage = language;
        com.android.internal.util.AnnotationValidations.validate(
                Deprecated.class, null, mLanguage);
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mLanguage);
        this.mLocale = locale;
        com.android.internal.util.AnnotationValidations.validate(
                NonNull.class, null, mLocale);
        this.mDataFormat = dataFormat;
        com.android.internal.util.AnnotationValidations.validate(
                DataFormat.class, null, mDataFormat);

        // onConstructed(); // You can define this method to get a callback
    }

    @DataClass.Generated.Member
    public static final @NonNull Parcelable.Creator<TranslationSpec> CREATOR
            = new Parcelable.Creator<TranslationSpec>() {
        @Override
        public TranslationSpec[] newArray(int size) {
            return new TranslationSpec[size];
        }

        @Override
        public TranslationSpec createFromParcel(@NonNull Parcel in) {
            return new TranslationSpec(in);
        }
    };

    @DataClass.Generated(
            time = 1620089623334L,
            codegenVersion = "1.0.23",
            sourceFile = "frameworks/base/core/java/android/view/translation/TranslationSpec.java",
            inputSignatures = "public static final  int DATA_FORMAT_TEXT\nprivate final @java.lang.Deprecated @android.annotation.NonNull java.lang.String mLanguage\nprivate final @android.annotation.NonNull android.icu.util.ULocale mLocale\nprivate final @android.view.translation.TranslationSpec.DataFormat int mDataFormat\n  void parcelLocale(android.os.Parcel,int)\nstatic  android.icu.util.ULocale unparcelLocale(android.os.Parcel)\nclass TranslationSpec extends java.lang.Object implements [android.os.Parcelable]\n@com.android.internal.util.DataClass(genEqualsHashCode=true, genHiddenConstDefs=true, genToString=true, genConstructor=false)")
    @Deprecated
    private void __metadata() {}


    //@formatter:on
    // End of generated code

}
