; APT (Automated Program Transformations) Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "ACL2")

(include-book "solve-templates")

(include-book "kestrel/event-macros/proof-preparation" :dir :system)
(include-book "kestrel/soft/defun-inst" :dir :system)
(include-book "kestrel/soft/defthm-inst" :dir :system)
(include-book "std/testing/must-succeed-star" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; This file contains generic proofs for
; the refinement theorems generated by SOLVE.
; The proofs are generic because they are based on the templates
; defined in solve-templates.lisp.
; The SOLVE implementation generates proofs
; according to the templates in this file.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Generate functions and theorems with proofs.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro gen-solution-for-rw-eqterm (&key vars)
  `(progn
     (defthmd rw-theorem
       (equal (m ,@vars (?f ,@vars))
              (equal (?f ,@vars)
                     (term ,@vars)))
       :hints (("Goal" :in-theory '(rwrule))))
     (gen-f :name f
            :inputs ,vars
            :body (term ,@vars)
            :guard t
            :guard-hints (("Goal" :in-theory nil)))
     (soft::defun-inst old-instance
       (old (?f . f)))
     (soft::defthm-inst rw-theorem-instance
       (rw-theorem (?f . f)))
     (defthmd sol-theorem
       (old-instance)
       :instructions
       ((:in-theory nil)
        :x
        (:rewrite rw-theorem-instance)
        (:prove :hints (("Goal" :in-theory '(f))))))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro gen-f-body-correct-with-proofs-for-rw-eqterm (&key vars)
  `(progn
     (defthmd lemma
       (equal (m ,@vars (?f ,@vars))
              (equal (?f ,@vars)
                     (term ,@vars)))
       :hints (("Goal" :in-theory '(rwrule))))
     (defthmd f-body-correct
       (implies (equal (?f ,@vars) (term ,@vars))
                (m ,@vars (?f ,@vars)))
       :hints (("Goal" :in-theory nil :use lemma)))))

(defmacro gen-f-body-correct-with-proofs-for-rw-t (&key vars)
  `(progn
     (defthmd lemma
       (equal (m ,@vars (?f ,@vars))
              t)
       :hints (("Goal" :in-theory '(rwrule))))
     (defthmd f-body-correct
       (implies (equal (?f ,@vars) nil)
                (m ,@vars (?f ,@vars)))
       :hints (("Goal" :in-theory nil :use lemma)))))

(defmacro gen-f-body-correct-with-proofs-for-sat-eqterm (&key vars)
  `(defthmd f-body-correct
     (implies (equal (?f ,@vars) (term ,@vars))
              (m ,@vars (?f ,@vars)))
     :hints (("Goal" :in-theory '(sat)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro gen-f-with-proofs-for-term (&key vars)
  `(gen-f :name f
          :inputs ,vars
          :body (term ,@vars)
          :guard t))

(defmacro gen-f-with-proofs-for-nil (&key vars)
  `(gen-f :name f
          :inputs ,vars
          :body nil
          :guard t))

(defmacro gen-new-with-proofs (&key vars)
  `(gen-new :name new
            :vars ,vars
            :?f ?f
            :f f
            :guard-hints (("Goal" :in-theory nil))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun gen-instantiation-1 ()
  '((x (old-witness))))

(defun gen-instantiation-2 ()
  '((x (mv-nth 0 (old-witness)))
    (y (mv-nth 1 (old-witness)))))

(defun gen-instantiation-3 ()
  '((x (mv-nth 0 (old-witness)))
    (y (mv-nth 1 (old-witness)))
    (z (mv-nth 2 (old-witness)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defmacro gen-old-if-new-with-proofs (instantiation)
  (let ((lemma-instance `(:instance f-body-correct ,@instantiation)))
    `(gen-old-if-new :name old-if-new
                     :old old
                     :new new
                     :hints (("Goal"
                              :in-theory '(old new-necc f)
                              :use ,lemma-instance)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Generate all the above functions and theorems with proofs.

(defun gen-all-with-proofs-for-rw-eqterm-fn (vars instantiation)
  `(progn
     (evmac-prepare-proofs)
     (gen-f-body-correct-with-proofs-for-rw-eqterm :vars ,vars)
     (gen-f-with-proofs-for-term :vars ,vars)
     (gen-new-with-proofs :vars ,vars)
     (gen-old-if-new-with-proofs ,instantiation)))

(defmacro gen-all-with-proofs-for-rw-eqterm (&key vars instantiation)
  `(make-event (gen-all-with-proofs-for-rw-eqterm-fn ',vars ,instantiation)))

(defun gen-all-with-proofs-for-rw-t-fn (vars instantiation)
  `(progn
     (evmac-prepare-proofs)
     (gen-f-body-correct-with-proofs-for-rw-t :vars ,vars)
     (gen-f-with-proofs-for-nil :vars ,vars)
     (gen-new-with-proofs :vars ,vars)
     (gen-old-if-new-with-proofs ,instantiation)))

(defmacro gen-all-with-proofs-for-rw-t (&key vars instantiation)
  `(make-event (gen-all-with-proofs-for-rw-t-fn ',vars ,instantiation)))

(defun gen-all-with-proofs-for-sat-eqterm-fn (vars instantiation)
  `(progn
     (evmac-prepare-proofs)
     (gen-f-body-correct-with-proofs-for-sat-eqterm :vars ,vars)
     (gen-f-with-proofs-for-term :vars ,vars)
     (gen-new-with-proofs :vars ,vars)
     (gen-old-if-new-with-proofs ,instantiation)))

(defmacro gen-all-with-proofs-for-sat-eqterm (&key vars instantiation)
  `(make-event (gen-all-with-proofs-for-sat-eqterm-fn ',vars ,instantiation)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Proof templates for n = 1.

(must-succeed*
 (gen-inputs-1-rw-eqterm)
 (gen-all-with-proofs-for-rw-eqterm :vars (x)
                                    :instantiation (gen-instantiation-1)))

(must-succeed*
 (gen-inputs-1-rw-t)
 (gen-all-with-proofs-for-rw-t :vars (x)
                               :instantiation (gen-instantiation-1)))

(must-succeed*
 (gen-inputs-1-sat-eqterm)
 (gen-all-with-proofs-for-sat-eqterm :vars (x)
                                     :instantiation (gen-instantiation-1)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Proof templates for n = 2.

(must-succeed*
 (gen-inputs-2-rw-eqterm)
 (gen-all-with-proofs-for-rw-eqterm :vars (x y)
                                    :instantiation (gen-instantiation-2)))

(must-succeed*
 (gen-inputs-2-rw-t)
 (gen-all-with-proofs-for-rw-t :vars (x y)
                               :instantiation (gen-instantiation-2)))

(must-succeed*
 (gen-inputs-2-sat-eqterm)
 (gen-all-with-proofs-for-sat-eqterm :vars (x y)
                                     :instantiation (gen-instantiation-2)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

; Proof templates for n = 3.

(must-succeed*
 (gen-inputs-3-rw-eqterm)
 (gen-all-with-proofs-for-rw-eqterm :vars (x y z)
                                    :instantiation (gen-instantiation-3)))

(must-succeed*
 (gen-inputs-3-rw-t)
 (gen-all-with-proofs-for-rw-t :vars (x y z)
                               :instantiation (gen-instantiation-3)))

(must-succeed*
 (gen-inputs-3-sat-eqterm)
 (gen-all-with-proofs-for-sat-eqterm :vars (x y z)
                                     :instantiation (gen-instantiation-3)))
